
//  (C) Copyright Edward Diener 2011
//  Use, modification and distribution are subject to the Boost Software License,
//  Version 1.0. (See accompanying file LICENSE_1_0.txt or copy at
//  http://www.boost.org/LICENSE_1_0.txt).

#if !defined(TTI_MEMBER_TYPE_HPP)
#define TTI_MEMBER_TYPE_HPP
  
#include <boost/config.hpp>
#include <boost/mpl/eval_if.hpp>
#include <boost/mpl/identity.hpp>
#include <boost/mpl/not.hpp>
#include <boost/type_traits/is_same.hpp>
#include <boost/preprocessor/cat.hpp>
#include <boost/tti/gen/member_type_gen.hpp>
#include <boost/tti/gen/namespace_gen.hpp>
#include <boost/tti/detail/dmem_type.hpp>
#include <boost/tti/detail/dnotype.hpp>

/*

  The succeeding comments in this file are in doxygen format.

*/

/** \file
*/

/// Expands to a metafunction whose typedef 'type' is either the named type or an unspecified marker type.
/**

    trait = the name of the metafunction within the tti namespace.
    
    name  = the name of the inner type.

    generates a metafunction called "trait" where 'trait' is the macro parameter.
    
              template<class TTI_T>
              struct trait
                {
                typedef unspecified type;
                };

              The metafunction types and return:
              
                TTI_T = the enclosing type.
                
                returns = 'type' is the inner type of 'name' if the inner type exists
                          within the enclosing type, else 'type' is an unspecified marker type.
                          
    The purpose of this macro is to encapsulate the 'name' type as the typedef 'type'
    of a metafunction, but only if it exists within the enclosing type. This allows for
    an evaluation of inner type existence, without generating a compiler error,
    which can be used by other metafunctions in this library.
    
*/
#define BOOST_TTI_TRAIT_MEMBER_TYPE(trait,name) \
    TTI_DETAIL_TRAIT_HAS_TYPE_MEMBER_TYPE(trait,name) \
    TTI_DETAIL_TRAIT_MEMBER_TYPE(trait,name) \
    template<class TTI_T> \
    struct trait : \
      boost::mpl::eval_if \
        < \
        BOOST_PP_CAT(trait,_detail)<TTI_T>, \
        BOOST_PP_CAT(trait,_detail_member_type)<TTI_T>, \
        boost::mpl::identity<BOOST_TTI_NAMESPACE::detail::notype> \
        > \
      { \
      }; \
/**/

/// Expands to a metafunction whose typedef 'type' is either the named type or an unspecified type.
/**

    name  = the name of the inner type.

    generates a metafunction called "member_type_name" where 'name' is the macro parameter.
    
              template<class TTI_T>
              struct member_type_name
                {
                typedef unspecified type;
                };

              The metafunction types and return:
              
                TTI_T = the enclosing type.
                
                returns = 'type' is the inner type of 'name' if the inner type exists
                          within the enclosing type, else 'type' is an unspecified marker type.
                          
    The purpose of this macro is to encapsulate the 'name' type as the typedef 'type'
    of a metafunction, but only if it exists within the enclosing type. This allows for
    an evaluation of inner type existence, without genrating a compiler error,
    which can be used by other metafunctions in this library.
    
*/
#define BOOST_TTI_MEMBER_TYPE(name) \
  BOOST_TTI_TRAIT_MEMBER_TYPE \
  ( \
  BOOST_TTI_MEMBER_TYPE_GEN(name), \
  name \
  ) \
/**/
  
namespace boost
  {
  namespace tti
    {
  
    /// A metafunction which checks whether the member 'type' returned from invoking the macro metafunction generated by BOOST_TTI_MEMBER_TYPE ( BOOST_TTI_TRAIT_MEMBER_TYPE ) is a valid type.
    /**

        template<class TTI_T>
        struct valid_member_type
          {
          static const value = unspecified;
          typedef mpl::bool_<true-or-false> type;
          };

        The metafunction types and return:

          TTI_T   = returned inner 'type' from invoking the macro metafunction generated by BOOST_TTI_MEMBER_TYPE ( BOOST_TTI_TRAIT_MEMBER_TYPE ).
      
          returns = 'value' is true if the type is valid, otherwise 'value' is false.
                          
    */
    template
      <
      class TTI_T
      >
    struct valid_member_type :
      boost::mpl::not_
        <
        boost::is_same
          <
          TTI_T,
          BOOST_TTI_NAMESPACE::detail::notype
          >
        >
      {
      };
      
    }
  }
  
#endif // TTI_MEMBER_TYPE_HPP
