
#pragma once
#ifndef NO_AUTOLINK_NADS_DDS_LIB 
#define NO_AUTOLINK_NADS_DDS_LIB
#endif


#include <memory>
#include <string>
#include <dds/sub/ddssub.hpp>
#include <dds/core/ddscore.hpp>

#include <dds/dds.hpp> 


#include <dds/pub/ddspub.hpp>
#include <rti/sub/SampleIterator.hpp>
#include <rti/util/util.hpp> // for sleep()

#include "NADS.hpp"
#include "ndds\dds_cpp\dds_cpp_domain.h"
#include "NadsDDSLib\NadsDDSLib.h"
#include "GenericSubscriber.h"

#include <thread>
#include <mutex>

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \namespace NADSDDS
///
/// \brief Namespace for Dealing DDS based communications
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace NADSDDS {
	using namespace std;
	using namespace dds::all;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CellPub
	///
	/// \brief Publication for Cell
	/// 	   
	/// This class handles the buffering scheme for DDS it holds a current value buffer
	/// and the topic and writer
	/// 
	/// \author Simop
	/// \date 4/29/2019
	///
	/// \tparam T DDS type, this comes from the DDS code generation
	////////////////////////////////////////////////////////////////////////////////////////////////////

	template<class T> class CellPub : public CellRoot {
	public:
		CellPub(std::string name, dds::domain::DomainParticipant& dm, size_t size, int id) :
			CellRoot(id), topic(dm, name)
		{
			writer = make_unique<dds::pub::DataWriter<T>>(dds::pub::Publisher(dm), topic);
			data.data().resize(size);
			data.eid() = _eid;
			_itemCount = size;
			_sizeInBytes = size * sizeof(T);
		}
		virtual void Send() override {
			auto ptr = writer.get();
			ptr->write(data);
		}
		virtual void* Raw() override {
			return (void*)data.data().data();
		}
		virtual int& eid() override {
			return data.eid();
		}
		unique_ptr< dds::pub::DataWriter<T> > writer;
		dds::topic::Topic<T> topic;
		T data;
	};





	//<singleton for holding domain participants
	class DDSDomainRepo {
	public:
		static dds::domain::DomainParticipant&  GetDomain(int id);
	private:
		typedef unique_ptr<DDSDomainRepo> TRef;
		typedef unique_ptr<dds::domain::DomainParticipant> TDomainRef;
		std::map<int, TDomainRef> _DomainRefs;
		static TRef _self;
		DDSDomainRepo();
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class DDSTopicInfo
	///
	/// \brief Information about the DDS topic.
	///
	/// A topic is basically a availible subscription. 
	/// 
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class DDSTopicInfo {
	public:
		typedef std::tuple<std::string, char> TTypeInfo;
		typedef std::vector<TTypeInfo> TTypeVect;
		static void Refresh();
		static const TTypeVect& GetTopics();
	private:
		DDSTopicInfo();
		void RefreshImpl();
		static DDSTopicInfo* Get();
		TTypeVect _info;
		typedef unique_ptr<DDSTopicInfo> TRef;
		static TRef _self;
	};

	//class CFloatPub : public CellPub<TypeFloat> {};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CFloatPub
	///
	/// \brief A float pub.
	/// 	    
	/// Provides Interface to Float data, hides DDS calls
	/// 
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CFloatPub {
	public:
		typedef unique_ptr<CellPub<TypeFloat>> TRef;
		CFloatPub(TRef &&in) :_ref(std::move(in)) {};
		CFloatPub(CFloatPub &&in) noexcept :_ref(std::move(in._ref)){}
		TRef _ref;
		CellPub<TypeFloat>* get() { return _ref.get(); }
		CellPub<TypeFloat>* operator->() { return _ref.get(); }
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CDoublePub
	///
	/// \brief A double pub.
	///
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CDoublePub {
	public:
		typedef unique_ptr<CellPub<TypeDouble>> TRef;
		CDoublePub(TRef &&in) :_ref(std::move(in)) {};
		CDoublePub(CDoublePub &&in) noexcept :_ref(std::move(in._ref)) {}
		TRef _ref;
		CellPub<TypeDouble>* get() { return _ref.get(); }
		CellPub<TypeDouble>* operator->() { return _ref.get(); }
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CCharPub
	///
	/// \brief A character pub.
	///
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CCharPub {
	public:
		typedef unique_ptr<CellPub<TypeChar>> TRef;
		CCharPub(TRef &&in) :_ref(std::move(in)) {};
		CCharPub(CCharPub &&in) noexcept :_ref(std::move(in._ref)) {}
		TRef _ref;
		CellPub<TypeChar>* get() { return _ref.get(); }
		CellPub<TypeChar>* operator->() { return _ref.get(); }
		CellPub<TypeChar>& operator*() const { return *_ref.get(); }
		CellPub<TypeChar>* operator()() { return _ref.get(); }
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CIntPub
	///
	/// \brief An int pub.
	///
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CIntPub {
	public:
		typedef unique_ptr<CellPub<TypeInt>> TRef;
		CIntPub(TRef &&in):_ref(std::move(in)) {};
		CIntPub(CIntPub &&in ) noexcept:_ref(std::move(in._ref)){}
		TRef _ref;
		CellPub<TypeInt>* get() { return _ref.get(); }
		CellPub<TypeInt>* operator->() { return _ref.get(); }
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CShortPub
	///
	/// \brief A short pub.
	///
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CShortPub {
	public:
		typedef unique_ptr<CellPub<TypeShort>> TRef;
		CShortPub(TRef &&in) :_ref(std::move(in)) {};
		CShortPub(CShortPub &&in) noexcept:_ref(std::move(in._ref)) {}
		operator CellPub<TypeShort>* () { return _ref.get(); }
		CellPub<TypeShort>* operator()() { return _ref.get(); }
		TRef _ref;
		CellPub<TypeShort>* get() { return _ref.get(); }
		CellPub<TypeShort>* operator->() { return _ref.get(); }
	};


	template < class T> class OnNewData {
	public:
		OnNewData() {};
		virtual void NewData(const T&) = 0;
	};

	typedef std::shared_ptr<OnNewData<TypeInt>>    TIntCB;
	typedef std::shared_ptr<OnNewData<TypeChar>>   TCharCB;
	typedef std::shared_ptr<OnNewData<TypeShort>>  TShortCB;
	typedef std::shared_ptr<OnNewData<TypeFloat>>  TFloatCB;
	typedef std::shared_ptr<OnNewData<TypeDouble>> TDoubleCB;

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class DataListener
	///
	/// \brief A data listener.
	/// 	   
	/// Implements a "no-op" reader/listener. This does not handle loss of data events, ect, it also
	/// provides a basic buffering scheme. The user can provide a call back class 
	/// \author Simop
	/// \date 4/29/2019
	///
	/// \tparam T Generic type parameter.
	////////////////////////////////////////////////////////////////////////////////////////////////////

	template < class T> class DataListener : public dds::sub::NoOpDataReaderListener<T> {
	public:
		typedef  std::shared_ptr<OnNewData<T>> TCbRef;
		DataListener(TCbRef ref) :_cb(ref),_hasData(false) {};
		void on_data_available(dds::sub::DataReader<T>& reader) {
			dds::sub::LoanedSamples<T> samples = reader.take();
			rti::sub::SampleIterator<T> it;// = dds::sub::LoanedSamples<T>::iterator
			for (rti::sub::SampleIterator<T>  sample_it = samples.begin(); sample_it != samples.end(); sample_it++) {
				if (sample_it->info().valid()) {
					_sample = sample_it->data();
					_hasData = true;
				}
			}
			if (_cb)
				_cb->NewData(_sample);
		}
		bool HasData() { return _hasData; }
		const T& GetCurrVal() { return _sample; }
	private:
		T _sample;
		bool _hasData;
		TCbRef _cb;
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class StreamBase
	///
	/// \brief base class for streams
	///
	/// \author Simop
	/// \date 4/30/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class StreamBase {
	public:
		StreamBase(dds::domain::DomainParticipant* dom, const std::string& topicName) :_dom(dom), _topicName(topicName) {};
	protected:
		dds::domain::DomainParticipant* _dom;
		std::string _topicName;
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class DataStream
	///
	/// \brief Handles the setting up the reader, and new data callback for the reader
	///
	/// Its really with the callback from our reader where most of our actual work is done. 
	/// This should likely be expanded in the future to properly setup the QOS settings, this is where
	/// we would handle setting up data coherence.
	/// 
	/// \author Simop
	/// \date 4/30/2019
	///
	/// \tparam T Generic type parameter.
	////////////////////////////////////////////////////////////////////////////////////////////////////

	template <class T> class DataStream : public StreamBase {
	public:
		typedef  std::shared_ptr<OnNewData<T>> TCbRef;
		DataStream(const std::string &topicName, dds::domain::DomainParticipant* dom, TCbRef ref) :StreamBase(dom, topicName), _listener(ref)
		{
			_topic = new dds::topic::Topic<T>(*_dom, topicName);
			_reader = new dds::sub::DataReader<T>(
				dds::sub::Subscriber(*_dom),
				*_topic,
				dds::core::QosProvider::Default().datareader_qos(),
				&_listener,
				dds::core::status::StatusMask::data_available());
		}
		~DataStream() {};
	private:
		std::string _subject;
		dds::sub::DataReader<T>* _reader;
		dds::topic::Topic<T>* _topic;
		DataListener<T> _listener;

	};


// BufferedCellRoot: public CellRoot{
// 	public:
// 		BufferedCellRoot(int id) : CellRoot(id) {} 

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CellSubsScription
	///
	/// \brief A cell subs scription.
	///
	/// \author Simop
	/// \date 4/29/2019
	///
	/// \tparam T DDS type like TypeFloat - which is auto generated code
	/// \tparam Y Base "C++" type, like float - which is viewable by host app
	////////////////////////////////////////////////////////////////////////////////////////////////////

	template <class T, class Y> class CellSubsScription : public BufferedCellRoot<T>{
	public:
		class CB : public OnNewData<Y> {
		public:
			CB():OnNewData<Y>(),_hasNewData(false){}
			virtual void NewData(const Y& valIn) override {
				lock_guard<std::mutex> lock(_mutex);
				_buffer = valIn;
				_hasNewData = true;
			}
			void GetData(std::vector<T>& valIn) {
				lock_guard<std::mutex> lock(_mutex);
				size_t bufferSize = _buffer.data().size();
				if (valIn.size() != bufferSize){
					valIn.resize(bufferSize);
				}
				for (size_t i = 0; i < bufferSize; i++) {
					valIn[i] = _buffer.data()[i];
				}
				if (std::is_same<T, char>::value){
					valIn.push_back(0);
				}
				_hasNewData = false;
			}
			bool HasNewData() {
				return _hasNewData;
			}
			std::string _name;
		protected:
			volatile bool _hasNewData;
			std::mutex _mutex;
			Y _buffer;
		};
		typedef 
		CellSubsScription(const std::string& name) :BufferedCellRoot<T>(name) {}
		void Register(){
			_cb = make_shared<CB>();
			
			_cb->_name = BufferedCellRoot<T>::_name;
			_dataStream = make_unique<DataStream<Y>>(BufferedCellRoot<T>::_name, &DDSDomainRepo::GetDomain(0), _cb);
		}
		virtual bool Swap() override {
			if (_cb && _cb->HasNewData()) {
				_cb->GetData(BufferedCellRoot<T>::_buffer);
				BufferedCellRoot<T>::_sizeInBytes = sizeof(T) * BufferedCellRoot<T>::_buffer.size();
				return true;
			}
			return false;
		}
		virtual int eid() override {
			return _streamBuffer.eid();
		}
	protected:
		Y _streamBuffer;
		unique_ptr<DataStream<Y>>  _dataStream;
		shared_ptr<CB> _cb;
		

	};
	

	typedef CellSubsScription<float, TypeFloat>  TFloatSubscription;
	typedef CellSubsScription<double,TypeDouble> TDoubleSubscription;
	typedef CellSubsScription<int,   TypeInt>    TIntSubscription;
	typedef CellSubsScription<short, TypeShort>  TShortSubscription;
	typedef CellSubsScription<char,  TypeChar>   TCharSubscription;
	
	typedef std::unique_ptr<TFloatSubscription> TFloatSubscriptionRef;
	typedef std::unique_ptr<TDoubleSubscription>TDoubleSubscriptionRef;
	typedef std::unique_ptr<TIntSubscription>   TIntSubscriptionRef;
	typedef std::unique_ptr<TShortSubscription> TShortSubscriptionRef;
	typedef std::unique_ptr<TCharSubscription>  TCharSubscriptionRef;

	/////////////////////////////////////////////////////////
	///\brief
	///    remove expired weak pointers from vector
	///\remark
	///    use:
	///      typedef weak_ptr<MyClass> TWeak
	///      typedef vector<TWeak> TVecs;
	///      TVecs vector;
	///      RemovedExpired<TVecs,TWeak>(vector); 
	////////////////////////////////////////////////////////
	template<class TVec, class TWp>
	void RemovedExpired(TVec &vec) {
		vec.erase(
			std::remove_if(
				vec.begin(),
				vec.end(),
				[](TWp cb) {return cb.expired(); }),
			vec.end());
	}

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CStatePriv
	///
	/// \brief Handles simulator state information
	/// 
	/// This class handles the simulator state information, and provides a callback for new frame.
	/// This call back is more or less suposed to take the place of RunSubsystem method for Scramnet
	/// based RTEX
	/// 
	/// \author Simop
	/// \date 4/29/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CStatePriv {
	public:
		static CStatePriv & Get();
		class Callback {
		public:
			Callback() {};
			virtual void OnFrame(int num) = 0;
		};

		typedef std::shared_ptr<Callback> TCBRef;
		typedef std::weak_ptr<Callback> TCBWRef;

		CStatePriv();
		int GetFrame();
		int GetState();
		void Register(TCBRef);
	protected:
		static std::unique_ptr<CStatePriv> _Ref;
		void Frame(int frameNumber);
		class OnNewFrame : public OnNewData<TypeInt> {
		public:
			OnNewFrame(CStatePriv* par):_par(par){}
			virtual void NewData(const TypeInt& val) override {
				_par->Frame(val.data()[0]);
			}
			CStatePriv* _par;
		};
		std::unique_ptr<DataStream<TypeInt>> _FrameSub;
		TIntSubscriptionRef _StateSub;
		typedef std::vector<TCBWRef> TCBVec;
		std::shared_ptr<OnNewFrame> _newFrameCb;
		TCBVec _Cbs;
		int _frame;
	};

}
