#pragma once


#include <NadsDDSLib/NadsDDSLib.h>
#include <Windows.h>

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \namespace NDDSClient
///
/// \brief Helper client classes for doing DDS/Gateway interface
////////////////////////////////////////////////////////////////////////////////////////////////////

namespace NDDSClient {
	class CellInput {
	public:
        CellInput() :_size(0),_target(nullptr),_type(0) {}
		size_t _size;
		void* _target;
		char _type;
		NADSDDS::CSubscription::TRef _sub;
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CCellInputMap
	///
	/// \brief Map of cell inputs.
	///
	/// \author Simop
	/// \date 4/26/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CCellInputMap {
	public:
		CCellInputMap();
		void RegInpElem(std::string name, void *_pBuffer, size_t _BufferSize, char _BufferType);
		void ReadInputElements();
	private:
		typedef std::unique_ptr<CellInput> TCellRef;
		std::map<std::string, TCellRef> _cellsIn;
	};
	class CellOutput {
	public:
		size_t _size;
		void* _target;
		char _type;
		NADSDDS::CPublication::TRef _sub;
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class CCellOutputMap
	///
	/// \brief Holds Map of cell outputs.
	/// Back end is a signlton, this can be used outside of DDSSimClient
	/// \author Simop
	/// \date 4/26/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class CCellOutputMap {
	public:
		CCellOutputMap() {}
		void RegInpElem(std::string name, void *_pBuffer, size_t _BufferSize, char _BufferType);
		void WriteOutputElements();
	private:
		typedef std::unique_ptr<CellOutput> TCellRef;
		std::map<std::string, TCellRef> _cellsOut;
	};

	////////////////////////////////////////////////////////////////////////////////////////////////////
	/// \class DDSSimClient
	///
	/// \brief The DDS simulation client.
	///
	/// \author Simop
	/// \date 4/26/2019
	////////////////////////////////////////////////////////////////////////////////////////////////////

	class DDSSimClient : public NADSDDS::CState::Callback {
	public:
		NADSDDS::CPublication::TRef _HeartBeat;
		NADSDDS::CPublication::TRef _StateAck;
		typedef std::shared_ptr<DDSSimClient> TRef; //this is shared with the DDS back end...so this must be a shared ptr;


		class StartUpCB {
		public:
			
			StartUpCB() {}
			virtual void Startup(DDSSimClient&)= 0;
		};
		typedef std::shared_ptr<StartUpCB> TStartCBRef;
		typedef std::weak_ptr<StartUpCB> TStartCBWRef;

		template<class T> class StartUpC : public StartUpCB {
		public:
			typedef std::shared_ptr<StartUpC<T>> TRef;
			static TRef Make(T* ptr) { return std::make_shared<StartUpC<T>>(ptr); }
			StartUpC(T* par) :_par(par) {}
			virtual void Startup(DDSSimClient& core) override{
				_par->Startup(core);
			}
			T* _par;
		};
		void RegisterStartup(TStartCBRef);

		class FrameCB {
		public:
			FrameCB() {}
			virtual int Frame(DDSSimClient&, int state, int frameNum) = 0;
		};
		typedef std::shared_ptr<FrameCB> TFrameCBRef;
		typedef std::weak_ptr<FrameCB> TFrameCBWRef;

		////////////////////////////////////////////////////////////////////////////////////////////////////
		/// \class Frame
		///
		/// \brief Template for receiving frame callbacks.
		/// use example:
		/// 
		/// DDSSimClient::Ref client;
		/// TFrameCBRef keep = TFrameCBRef(new DDSSimClient::Frame<MyClass>(this));
		/// client->RegisterFrameCB(keep);
		/// 
		/// \author Simop
		/// \date 4/26/2019
		///
		/// \typeparam T Host class to receive the callback.
		////////////////////////////////////////////////////////////////////////////////////////////////////

		template<class T> class NewFrame : public FrameCB {
		public:
			typedef std::shared_ptr<NewFrame<T>> TRef;
			static TRef Make(T* ptr) { return std::make_shared<NewFrame<T>>(ptr); }
			NewFrame(T* par) :_par(par) {}
			virtual int Frame(DDSSimClient& core, int state, int frameNum)  override {
				return _par->Frame(core,state,frameNum);
			}
			T* _par;
		};
	

		void RegisterFrameCB(TFrameCBRef);
		virtual void OnFrame(int frame);
		static TRef Make(TFrameCBRef, TStartCBRef, const std::string& name);

		//char
		void RegInputCell(const std::string& name, char* pnt, size_t cnt) {
			RegInputCell(name, pnt, cnt * sizeof(char), 'c');
		}
		void RegInputCell(const std::string& name, char& pnt) {
			RegInputCell(name, &pnt, sizeof(char), 'c');
		}

		//short
		void RegInputCell(const std::string& name, short* pnt, size_t cnt) {
			RegInputCell(name, pnt, cnt * sizeof(short), 's');
		}
		void RegInputCell(const std::string& name, short& pnt) {
			RegInputCell(name, &pnt, sizeof(short), 's');
		}

		//int
		void RegInputCell(const std::string& name, int* pnt, size_t cnt) {
			RegInputCell(name, pnt, cnt * sizeof(int), 'i');
		}
		void RegInputCell(const std::string& name, int& pnt) {
			RegInputCell(name, &pnt, sizeof(int), 'i');
		}

		//float
		void RegInputCell(const std::string& name, float* pnt, size_t cnt) {
			RegInputCell(name, pnt, cnt * sizeof(float), 'f');
		}
		void RegInputCell(const std::string& name, float& pnt) {
			RegInputCell(name, &pnt, sizeof(float), 'f');
		}

		//double
		void RegInputCell(const std::string& name, double* pnt, size_t cnt) {
			RegInputCell(name, pnt, cnt * sizeof(double), 'd');
		}
		void RegInputCell(const std::string& name, double& pnt) {
			RegInputCell(name, &pnt, sizeof(double), 'd');
		}

		//char
		void RegOutputCell(const std::string& name, char* pnt, size_t cnt) {
			RegOutputCell(name, pnt, cnt * sizeof(char), 'c');
		}
		void RegOutputCell(const std::string& name, char& pnt) {
			RegOutputCell(name, &pnt, sizeof(char), 'c');
		}

		//short
		void RegOutputCell(const std::string& name, short* pnt, size_t cnt) {
			RegOutputCell(name, pnt, cnt * sizeof(short), 's');
		}
		void RegOutputCell(const std::string& name, short& pnt) {
			RegOutputCell(name, &pnt, sizeof(short), 's');
		}

		//int
		void RegOutputCell(const std::string& name, int* pnt, size_t cnt) {
			RegOutputCell(name, pnt, cnt * sizeof(int), 'i');
		}
		void RegOutputCell(const std::string& name, int& pnt) {
			RegOutputCell(name, &pnt, sizeof(int), 'i');
		}

		//float
		void RegOutputCell(const std::string& name, float* pnt, size_t cnt) {
			RegOutputCell(name, pnt, cnt * sizeof(float), 'f');
		}
		void RegOutputCell(const std::string& name, float& pnt) {
			RegOutputCell(name, &pnt, sizeof(float), 'f');
		}

		//double
		void RegOutputCell(const std::string& name, double* pnt, size_t cnt) {
			RegOutputCell(name, pnt, cnt * sizeof(double), 'd');
		}
		void RegOutputCell(const std::string& name, double& pnt) {
			RegOutputCell(name, &pnt, sizeof(double), 'd');
		}
	private:
		NADSDDS::CState _dds_state;
		DDSSimClient();
		void StartUp();
		void RegInputCell(const std::string& name, void* pnt, size_t size, char type);
		void RegOutputCell(const std::string& name, void* pnt, size_t size, char type);
		std::vector<TFrameCBWRef> _FrameCBs;
		std::vector<TStartCBWRef> _StartupCBs;
		CCellInputMap _map;
		CCellOutputMap _omap;
		std::string _name;
		int _currentFrame;
		bool _Started;
	};
}