/////////////////////////////////////////////////////////////////////////////
//
// (C) Copyright 2020 by National Advanced Driving Simulator and
// Simulation Center, The University of Iowa. All rights reserved.
//
//
/////////////////////////////////////////////////////////////////////////////
#include <string>
#include <vector>
#include <map>
#include <memory>
#include "DaqPlayer.h"
using namespace std;

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn DaqPlayer::DaqPlayer(int stride)
///
/// \brief  Constructor
///
/// \author D A Heitbrink
/// \date   10/9/2020
///
/// \param  stride  How many frames per interation we advance in the playback
////////////////////////////////////////////////////////////////////////////////////////////////////

DaqPlayer::DaqPlayer(int stride):
    _stride(stride),
    _firstFrame(-1),
    _lastFrame(-1),
    _currentFrame(-1),
    _DaqFrameRate(-1) 
{
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn bool DaqPlayer::OpenDaqFile(const std::string& fileName,TStrVec items)
///
/// \brief  Opens daq file, may block for a long time on the first time opening a file
/// 
/// 
/// \author D A Heitbrink
/// \date   10/9/2020
///
/// \param  fileName    Filename of the file.
/// \param  items       list of cells to read every frame,
///
/// \returns    True if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

bool DaqPlayer::OpenDaqFile(const std::string& fileName,TStrVec items) {
    if (_pDaqIo) {
        _pDaqIo.reset();
    }
    _cellList = items;
    string temp;
    _pDaqIo =  std::make_unique<CDaqLowLevelIo>();
    struct stat daqTocInfo;
    //_pDaqIo->SetCallbacks(NULL, &DaqProgressCallBack);
        temp = fileName;
        temp += ".toc";
        //see if we have a .toc file, if not create one
        if (0 != stat(temp.c_str(), &daqTocInfo)) {
            if (!_pDaqIo->Open(fileName)) {
                string message = "Unable to load DAQ file - ";
                message += _pDaqIo->GetLastError();
                _pDaqIo.reset();
                return false;
            }
            if (!_pDaqIo->CheckIntegrity(true, true)) {
                _pDaqIo.reset();
                return false;
            }
        }
        else if (!_pDaqIo->Open(fileName) || !_pDaqIo->TocFileExists()) {
            string message = "Unable to load DAQ file - ";
            message += _pDaqIo->GetLastError();
            return false;
        }
        _pDaqIo->GetChannels(_allDaqChannels);
        
        _DaqFrameRate = _pDaqIo->GetFrequency();
        if (_DaqFrameRate < 60)
            _DaqFrameRate = 60;
        size_t cnt = 0;
        for (TChanInfoVec::iterator i = _allDaqChannels.begin(); i != _allDaqChannels.end(); i++,cnt++) {
            _varNameList.push_back(i->GetName());
            _nameToChannel[i->GetName()] = cnt;
            for (auto & val : _cellList) {
                if (i->GetName() == val) {
                    _getDaqChannels.push_back(i->GetId());
                    _daqNameToIndex[val] = (int)_getDaqChannels.size() - 1;
                }
            }
        }
        CDaqBuffer empty;
        _daqBuffers.resize(_getDaqChannels.size(), empty);
        TConvertOptions opt = eNO_OPTION;


        opt = opt + eFILL_MISSING;
        _firstFrame =_pDaqIo->GetFirstFrame();
        _lastFrame = _pDaqIo->GetFrames() / (_DaqFrameRate / 60) + _firstFrame;
        //lets see if we have any droped frames
        _currentFrame = _firstFrame;


        _pDaqIo->QueryIntegrityValues(
            _stats.drops, _stats.dropedFrm, _stats.skips, _stats.eofStat);
        return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn DaqPlayer& DaqPlayer::operator++()
///
/// \brief  Increments to the next frame in replay
///
/// This functions takes current frame + stride, set EOF if this has advanced to the last frame
/// it throws if past last frame;
/// 
/// \author D A Heitbrink
/// \date   10/9/2020
///
/// \exception  DaqPlayer::EndOfFile    Thrown when an read past end of file.
/// \exception  DaqPlayer::DaqReadError Thrown when an file read error occurs.
///
/// \returns    The result of the operation.
////////////////////////////////////////////////////////////////////////////////////////////////////

DaqPlayer& DaqPlayer::operator ++() {
    _currentFrame += _stride;
    if (_currentFrame > _lastFrame) {
        throw DaqPlayer::EndOfFile();
    }
    if (!GotoFrame(_currentFrame)) {
        throw DaqPlayer::DaqReadError(_pDaqIo->GetLastError().c_str());
    }
    return *this;

}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn bool DaqPlayer::Eof() const
///
/// \brief  is current frame > last frame
///
/// \author D A Heitbrink
/// \date   10/9/2020
///
/// \returns    True if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

bool DaqPlayer::Eof() const {
    return _currentFrame > _lastFrame;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn bool DaqPlayer::GotoFrame(int frameNum)
///
/// \brief  Advance current frame to input, reads data for the said frame
///
/// \author D A Heitbrink
/// \date   10/9/2020
///
/// \param  frameNum    The frame number.
///
/// \returns    True if it succeeds, false if it fails.
////////////////////////////////////////////////////////////////////////////////////////////////////

bool DaqPlayer::GotoFrame(int frameNum) {
    
    //DAH this is +3, we are counting by 4's and we are not guarented to have data every 4th frame, although this is true 95% of the time 
    if (!_pDaqIo->GetFullData(&_getDaqChannels, _daqBuffers, eFILL_MISSING, frameNum, frameNum + _stride - 1,0,_stride)) {
        return false;
    }
    _currentFrame = frameNum;
    return true;
}

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn bool DaqPlayer::AddCell(const std::string& str)
///
/// \brief  Adds a cell to list to read
///
/// \author D A Heitbrink
/// \date   10/9/2020
///
/// \param  str The string.
///
/// \returns    False if the cell does not exist in the DAQ file
////////////////////////////////////////////////////////////////////////////////////////////////////

bool DaqPlayer::AddCell(const std::string& str) {
    auto itr = _nameToChannel.find(str);
    if (itr == _nameToChannel.end()) {
        return false;
    }
    //see if we have it already
    if (_daqNameToIndex.find(str) == _daqNameToIndex.end())
        return true;
    int id = _allDaqChannels[(itr->second)].GetId();
    _getDaqChannels.push_back(id);
    _daqNameToIndex[str] = (int)_getDaqChannels.size()-1;
    return true; 
    
}
optional<CDaqChannelInfo> DaqPlayer::GetChannelInfo(const string& str) {
    optional<CDaqChannelInfo> ret;
    auto itr = _nameToChannel.find(str);
    if (itr == _nameToChannel.end()) {
        return optional<CDaqChannelInfo>();
    }
    return optional<CDaqChannelInfo>(_allDaqChannels[itr->second]);
}