/////////////////////////////////////////////////////////////////////////////
//
// (C) Copyright 2005-2020 by National Advanced Driving Simulator and
// Simulation Center, The University of Iowa. All rights reserved.
//
//
/////////////////////////////////////////////////////////////////////////////
#pragma once

#include <DaqIoLib.h>
#include <optional>
class DaqPlayer
{
public:
    typedef unsigned char            TDaqByte;
    typedef std::vector<int>         TIntVec;
    typedef std::vector<std::string> TStrVec;
    typedef std::vector<CDaqBuffer>  TDataVec;
    typedef std::vector<CDaqChannelInfo> TChanInfoVec;
    typedef std::unique_ptr<CDaqLowLevelIo> TDaqFile;
    struct TStats{
        TStats() :eofStat(eDAQ_EOF_UNDEFINED){}
        TIntVec drops;
        TIntVec dropedFrm;
        TIntVec skips;
        TDaqEofStatus eofStat;
    };    
    class EndOfFile:public std::exception {
    public:
        EndOfFile() {}
    };

    class DaqReadError:public std::exception {
    public:
        DaqReadError(char const* const Message):std::exception(Message) {}
    };

    DaqPlayer(int stride = 4);
    DaqPlayer& operator ++();

    std::optional<CDaqChannelInfo> GetChannelInfo(const std::string&);

    bool OpenDaqFile(const std::string &file,TStrVec items);
    bool GotoFrame(int frameNum);
    bool Eof() const;
    const TStats& GetStats() const {return _stats;}
    virtual void DaqReadProgress(const char *name, int pram1, int param2) {};
    virtual void DaqError(const char *, const char *) {};
    const std::string& GetLastError() const { return m_lastError; };
    bool AddCell(const std::string&);
    int GetCurrentFrame() { return _currentFrame; };
    template <class T> std::vector<T> GetData(const std::string& name);

private:
    TDaqFile         _pDaqIo;
    TChanInfoVec     _allDaqChannels;
    struct TDifferntialDataFrame {
        TDifferntialDataFrame() :frame(-1) {};
        TDifferntialDataFrame(const TDifferntialDataFrame& in):frame(in.frame), data(in.data) {};
        TDifferntialDataFrame(TDifferntialDataFrame&& in) noexcept:frame(in.frame), data(std::move(in.data))  {};
        int frame;
        CDaqBuffer data;
    };
    
    TIntVec          _getDaqChannels;
    CDaqLowLevelIo::TProgressHandlerRef _prog;
    CDaqLowLevelIo::TErrorHandlerRef _error;
    int _stride;
    TStrVec _cellList;
    std::map<std::string,int>  _daqNameToIndex;
    std::vector<CDaqBuffer>  _daqBuffers;
    std::map<std::string, size_t> _nameToChannel;

    TStrVec _varNameList;
    int _DaqFrameRate;
    int _firstFrame;
    int _lastFrame;
    int _currentFrame;
    TStats _stats;
    TDaqFile file;
    std::string m_lastError;
};

////////////////////////////////////////////////////////////////////////////////////////////////////
/// \fn template <class T> std::vector<T> DaqPlayer::GetData(const std::string& name)
///
/// \brief  Gets a data for a cell
///         
/// Gets data for a cell, does type casting from cell type to type T
/// 
/// 
/// \tparam T   type parameter, must be able to cast from numeric type.
/// \param  name    The name of the cell.
///
/// \returns    The data.
////////////////////////////////////////////////////////////////////////////////////////////////////

template <class T> std::vector<T> DaqPlayer::GetData(const std::string& name){
    std::vector<T> ret;
    auto idx = _daqNameToIndex.find(name);
    if (idx == _daqNameToIndex.end())
        return ret;

    auto index = idx->second;
    auto type = _allDaqChannels[index].GetType();

    int typeSize = sizeof(T);
    if (index >= _daqBuffers.size())
        return ret;
    auto valValueVec = _daqBuffers.at(index);
    auto cnt = valValueVec.GetCount();

    ret.reserve(cnt);

    for (size_t i = 0; i < cnt; i++) {
        if (type == 'i') {
            ret.push_back((T)*valValueVec.GetInt((int)i));
        }
        else if (type == 's'){
            ret.push_back((T)*valValueVec.GetShort((int)i));
        }
        else if (type == 'c') {
            ret.push_back((T)*valValueVec.GetChar((int)i));
        }
        else if (type == 'f') {
            ret.push_back((T)*valValueVec.GetFloat((int)i));
        }
        else if (type == 'd') {
            ret.push_back((T)*valValueVec.GetDouble((int)i));
        }
    }
    return ret;
}

