#include <stdio.h>
#include <string>
#include <vector>
#include <map>

using namespace std;

class CDaqBuffer;
class CDaqChannelInfo;


#define DAQ_MAGIC_NUM_VER_2_0 0x2C3D4E5F
#define DAQ_MAGIC_NUM_VER_2_1 0x2C3D4E6F
#define DAQ_MAGIC_NUM_VER_2_2 0x2C3D4E7F
#define DAQ_END_MARK          0x1A2B2C3D
#define DAQ_TOC_MAGIC_NUM     0x2C3ABCDE

typedef unsigned char       TDaqByte;
typedef vector<int>         TIntVec;
typedef vector<string>      TStrVec;
typedef vector<CDaqBuffer>  TDataVec;
typedef vector<FILE *>      TFileVec;
typedef vector<CDaqChannelInfo> TChanInfoVec;


const int cMaxDaqRec   = 128*1024;   ///< maximum size of data per DAQ record
const int cMaxChannels = 500;       ///< maximum number of channels we can handle

///////////////////////////////////////////////////////////////////////////////////
///
/// This enumeration encapsulates various conversion options
/// Appropriate operators have been defined so combining options
/// can be done by using + or |
///
typedef enum {
	eNO_OPTION            = 0,
	eEXPAND_DIFFERENTIAL  = 1,      
	eFILL_MISSING         = 2,      
	eINCLUDE_FRAME        = 4,     
	eBINARY_FORMAT        = 32,     
	eASCII_FORMAT         = 64
} TConvertOptions;

inline TConvertOptions
operator+(TConvertOptions left, TConvertOptions right) {
	return (TConvertOptions)((int)left + (int)right);
}

inline TConvertOptions
operator|(TConvertOptions left, TConvertOptions right) {
	return (TConvertOptions)((int)left | (int)right);
}

///////////////////////////////////////////////////////////////////////////////////////////
///
/// Return code used to specify the status of the end marker of the DAQ file
///
typedef enum {
	eDAQ_EOF_OK = 0,			///< Found full EOF marker
	eDAQ_EOF_PARTIAL_MARK = 1,	///< Found a partial marker
	eDAQ_EOF_NOTFOUND = 2,		///< No EOF marker found
	eDAQ_EOF_UNDEFINED = 3		///< No information available yet
} TDaqEofStatus;

///////////////////////////////////////////////////////////////////////////////////////////
///
/// This class represents all information about a channel stored in the DAQ file. 
/// Each channel consist of a number of samples, each sample containing one or more 
/// values of the same type. The term item is used to refer to one of those values. 
/// The term record is used to refer to all the values in a sample.
///
/// The information maintained for each channel includes an identifier (which 
/// is nothing more than the ordinal number of the channel in the definition array), 
/// the type of the data, the cardinality of the data, its name and the capture 
/// rate. The capture rate is given in Hz and is always a sub-multiple of the DAQ 
/// maximum sampling rate.
///
/// In addition, there are three options that refer to the method by which the 
/// data for a channel is collected. The sampling rate indicates how often 
/// data for that channel is sampled. Usually, the maximum sampling rate 
/// is 240 Hz, but most channels are sampled at a frequency lower than that. 
/// Data stored in 'differential' mode are only stored when their value has 
/// changed from the previously stored value. This can save a tremendous 
/// amount of space especially for large but largely unchanged channels. 
/// Finally, variable length channels are array channels (i.e., cardinality > 1) 
/// that have only a subset of the array stored for each sample. This is done 
/// to save space, when the real-time software knows that only a subset of the 
/// array has valid data. 
///
class CDaqChannelInfo {
public:
	CDaqChannelInfo() {
		m_Id = m_Items = m_CapRate = m_Type = -1;
		m_VarLen = false;
		m_Name = "";
	}
	~CDaqChannelInfo() {};
    int				GetItemSize()	const;
	int				GetRecSize()	const;
	int				GetItemCount()	const;
	int				GetType()		const;
	int				GetId()			const;
	const string&	GetName()		const;
	int				GetCapRate()	const;
	bool			IsVarLen()		const;

	friend class CDaqLowLevelIo;

private:
	int		m_Id;
	int		m_Items;
	string	m_Name;
	int		m_CapRate;
	int		m_Type;
	bool	m_VarLen;
};


/// Return true, if channel is a variable length array
inline bool
CDaqChannelInfo::IsVarLen(void) const
{
	return m_VarLen;
}


///// Return sampling rate, in Hz
inline int
CDaqChannelInfo::GetCapRate(void) const
{
	return m_CapRate;
}

///// Return name as specified in cell file
inline const string &
CDaqChannelInfo::GetName(void) const
{
	return m_Name;
}

/// 
/// Return the channel identifier, a small integer that can be
/// used to reference the channel in other calls.
///
/// Keep in mind that the channel identifer can change among DAQ files,
/// even if it refers to the same cell variable.
///
inline int
CDaqChannelInfo::GetId(void) const
{
	return m_Id;
}

///// Return type (c=char, f=float, i=int, s=short, d=double
inline int
CDaqChannelInfo::GetType(void) const
{
	return m_Type;
}

///// Return number of items in each sample
inline int
CDaqChannelInfo::GetItemCount(void) const
{
	return m_Items;
}

/////////// Return record size, in bytes
inline int
CDaqChannelInfo::GetRecSize(void) const
{
	return GetItemSize() * GetItemCount(); 
}


/////////////////////////////////////////////////////////////////////////////////////////
///
/// This support class provides buffer space for copying daq data out of a 
/// DAQ file. It is used in lieu of raw buffers to minimize the likelihood of 
/// memory leaks.
///
/// The class itself is not aware of the type or cardinality of the data, 
/// however, it provides a few type-specific access functions to facilitate 
/// data access when the user knows the type of data.
///
class CDaqBuffer {
public:
	CDaqBuffer() { m_Count = 0; };
	~CDaqBuffer() {};

	vector<TDaqByte>::const_pointer GetDataPtr() const { return &m_Data[0]; };

	int             GetCount() const ///< Return number of typed elements in buffer
			{ return m_Count; };

	int             GetSize() const ///< Return buffer size in bytes
			{ return (int)m_Data.size(); };

	char*   GetChar(int ind=0)     ///< Return char type pointer to data
			{ return ((char *)&m_Data[0]) + ind; };

	int*    GetInt(int ind=0)       ///< Return int type pointer to data
			{ return ((int *)&m_Data[0]) + ind; };

	float*  GetFloat(int ind=0)     ///< Return float type pointer to data
			{ return ((float *)&m_Data[0]) + ind; };

	double* GetDouble(int ind=0)    ///< Return double type pointer to data
			{ return ((double *)&m_Data[0]) + ind; };

	short*  GetShort(int ind=0)     ///< Return short type pointer to data
			{ return ((short *)&m_Data[0]) + ind; };

	friend class CDaqLowLevelIo;
	vector<TDaqByte>    m_Data;     // data is stored here

protected:
	void Append(const void *pData, int size, int count);
	void Replace(const void *pData, size_t size, int count);
	void AllocSpace(int     size, int count=0);
	void Clear(void) { m_Data.clear(); m_Count = 0; };

	int                 m_Count;    // count of data, not in bytes but in type
};


/////////////////////////////////////////////////////////////////////////////////
///
/// This function adds data to the DaqBuffer.  It is used by the DAQ library
/// to build the buffers returned to the user, and as such, is not meant to
/// be used by the library client.
///
/// The function receives a pointer to the data, the size of the buffer in 
/// bytes and a count of elements in that buffer.  The class deals with
/// untyped buffers, so the only way to keep track of how many elements
/// are in the buffer (i.e., 2 integers as opposed to 8 bytes) is for the
/// caller to provide that information, which in this case is the count 
/// argument.
///
inline void
CDaqBuffer::Append(
	const void*  p,      ///< pointer to data to append
	int		     size,   ///< size in bytes of data
	int          count)  ///< count of data items in that buffer
{
	size_t oldsize  = m_Data.size();

	m_Data.resize(oldsize + size);
	memcpy(&m_Data[0]+oldsize, p, size);
	
	m_Count += count;
}

/////////////////////////////////////////////////////////////////////////////////
///
/// This function replaces the existing data with new one. 
///
/// The function receives a pointer to the data, the size of the buffer in 
/// bytes and a count of elements in that buffer.  The class deals with
/// untyped buffers, so the only way to keep track of how many elements
/// are in the buffer (i.e., 2 integers as opposed to 8 bytes) is for the
/// caller to provide that information, which in this case is the count 
/// argument.
///
inline void
CDaqBuffer::Replace(
	const void*  p,      ///< pointer to data to append
	size_t	     size,   ///< size in bytes of data
	int          count)  ///< count of data items in that buffer
{
	size_t oldsize  = m_Data.size();

	if ( oldsize < size ) {
		m_Data.resize(size);
	}
	memcpy(&m_Data[0], p, size);
	
	m_Count = count;
}


/////////////////////////////////////////////////////////////////////////////////
///
/// This function allocates space in the DaqBuffer.  
///
///
inline void
CDaqBuffer::AllocSpace(
	int     size,   ///< size in bytes of data
	int     count)  ///< count of data items in that buffer
{
	m_Data.resize(size);
	m_Count = count;
}


/////////////////////////////////////////////////////////////////////////////
///
/// This class provides an API that does low level reading from a DAQ 
/// file. In particular, each DAQ file is treated as a set of independent 
/// channels, with each channel having its own type/cardinality/sampling 
/// rate and sampling method.
///
/// As a low level API, it only provides data integrity and raw data 
/// reading capability for either the whole daq file, or for selected 
/// channels. Except for these options, there are no other data conditioning 
/// options or partitioning options. 
///
class CDaqLowLevelIo {
public:
	CDaqLowLevelIo();
	~CDaqLowLevelIo();

	bool           Open(const string &);
	void           GetChannels(TChanInfoVec& channels) const;
	bool           GetChannelInfo(int chId, CDaqChannelInfo &info) const;
	int            GetFirstFrame() const;
	int            GetLastFrame() const;  
	int            GetFrames() const;
	int            GetFrequency() const;
	const string&  GetDate() const;
	const string&  GetVersion() const;
	const string&  GetTitle() const;
	const string&  GetSubject() const;
	const string&  GetRun() const;
	const string&  GetRunInst() const;
	const string&  GetFileName() const { return m_Filename; };
	size_t         GetNumChannels() const;
	bool           TocFileExists() const { return m_TocExists; };

	bool CheckIntegrity(bool buildToc = false, bool forceFileCreate = false);
	bool QueryIntegrityValues(TIntVec& drops, TIntVec& dropedFrm, TIntVec& skips, 
			TDaqEofStatus &stat);
	const string& GetLastError(void) const 
			{ return m_LastError; };
	void Close(void);

	bool FrameDropped(int frm);
	bool FrameHasData(int frm);

	void SetCallbacks( void errf(const char *, const char *), void progf(const char *, int, int));

	bool LowLevelDataRead(const TIntVec* chans, TDataVec& data, vector<TIntVec>* pFrameList=0);
	bool LowLevelDataCopy(const TIntVec* chans, TFileVec& dest, 
				TConvertOptions options = eBINARY_FORMAT + eINCLUDE_FRAME);
	bool GetFullData(const TIntVec* chans, TDataVec& data, TConvertOptions option = eNO_OPTION,
				int firstFrame = -1, int lastFrame = -1, vector<TIntVec>* pFrameList = 0);

private:
	enum TReadErrorCode {
		eDAQ_READ_OK,			
		eDAQ_READ_ERROR,		// read error (short file, premature eof etc.
		eDAQ_READ_EOF,			// found end marker
		eDAQ_READ_FAILED,		// lost sync
		eDAQ_READ_BAD_DATA,		// bad data and couldn't recover
		eDAQ_NO_SUCH_FRAME,		// asked to read a frame that doesn't exist in TOC
		eDAQ_READ_INTERNAL_ERROR
	};

	enum TResyncOutcome {
		eRESYNC_CHANNEL,	// resynced ok, next thing to read is channel
		eRESYNC_FRAME,		// resynced ok, next thing to read is frame
		eRESYNC_NONE		// could not resync
	};

	bool ReadHeader(FILE *);
	bool ReadChannelInfo(FILE *);
	bool DetectFrameHeader(int recFrame);
	bool ResyncFrame(int curFrame, int& skipCount);
	TResyncOutcome ResyncChannel(int recFrame, int& skipCount);
	bool WriteTocFile(const string& fname);
	bool ReadTocFile(const string& fname);
	TReadErrorCode ReadFrameHeader(int,	int& ,int&, int&);
	TReadErrorCode ReadDataForOneFrame(int, TIntVec, TDataVec &, void *, vector<bool>&);

//	bool GenPurpTraverse(bool frameCB(int frm, int numC, bool end), 
//							bool channelCB(int frm, int chan) );

	string	m_Filename;
	string  m_Version;			// file version
	FILE*	m_pFile;
	string	m_Title;			// Simulation title, as stored in DAQ file
	string	m_Date;				// Date of collection, as stored in DAQ file
	string	m_Subj;				// As stored in DAQ file
	string	m_Run;				// As stored in DAQ file
	string	m_RunInst;			// As stored in DAQ file
	int		m_NumEntries;		// As stored in DAQ file
	int		m_Frequency;		// As stored in DAQ file

	void	(*m_UserErrorFunc)(const char *, const char *);
	void	(*m_UserProgrFunc)(const char *, int, int);

	bool                 m_TocExists;	// Indicates a TOC exists in a file
	bool				 m_HaveToc;		// Indicates if we have TOC in memory
	TChanInfoVec		 m_Channels;	// Channel information
	long				 m_DataOffset;	// where data starts in the file 
	map<int, int>        m_Toc;			// given a sim frame, gives the offset in the file
	vector<int>	         m_DroppedFrames;
	int					 m_FirstFrame;
	int					 m_LastFrame;
	mutable string		 m_LastError;
	TDaqEofStatus        m_EofStatus;

	// These are for data integrity
	TIntVec				m_FrameDrops;		// sequences of skipped frames
	TIntVec				m_Skips;			// groups of bytes skipped
};

inline void
CDaqLowLevelIo::GetChannels(TChanInfoVec & chn) const
{
	chn = m_Channels;
}

/// Return the title stored in the DAQ file
inline const string &
CDaqLowLevelIo::GetTitle(void) const
{
	return m_Title;
}


/// Return the DAQ file version number
inline const string &
CDaqLowLevelIo::GetVersion(void) const
{
	return m_Version;
}

/// Return the run instance name.  If the feature is not
/// supported due to an older version, an empty string is returned
inline const string &
CDaqLowLevelIo::GetRunInst(void) const
{
	return m_RunInst;
}

/// Return the subject name
inline const string &
CDaqLowLevelIo::GetSubject(void) const
{
	return m_Subj;
}

/// Return the name of the Run that produced the daq file
inline const string &
CDaqLowLevelIo::GetRun(void) const
{
	return m_Run;
}

/// Return the file creation date as stored in the heade
inline const string &
CDaqLowLevelIo::GetDate(void) const
{
	return m_Date;
}


/// Return the sampling frequency
inline int
CDaqLowLevelIo::GetFrequency(void) const
{
	return m_Frequency;
}

/// Return the total number of frames in the file
inline int
CDaqLowLevelIo::GetFrames(void) const
{
	return m_LastFrame - m_FirstFrame;
}

/// Return the first frame number
inline int
CDaqLowLevelIo::GetFirstFrame(void) const
{
	return m_FirstFrame;
}

/// Return the last frame number
inline int
CDaqLowLevelIo::GetLastFrame(void) const
{
	return m_LastFrame;
}

/// This funcion returns the number of channels in the DAQ file
inline size_t
CDaqLowLevelIo::GetNumChannels(void) const
{
	return m_Channels.size();
}