
///////////////////////////////////////////////////////////////////////////////////////////
///
/// \ingroup clib
/// This class represents all information about a channel stored in the DAQ file. 
/// Each channel consist of a number of samples, each sample containing one or more 
/// values of the same type. The term item is used to refer to one of those values. 
/// The term record is used to refer to all the values in a sample.
///
/// The information maintained for each channel includes an identifier (which 
/// is nothing more than the ordinal number of the channel in the definition array), 
/// the type of the data, the cardinality of the data, its name and the capture 
/// rate. The capture rate is given in Hz and is always a sub-multiple of the DAQ 
/// maximum sampling rate.
///
/// In addition, there are three options that refer to the method by which the 
/// data for a channel is collected. The sampling rate indicates how often 
/// data for that channel is sampled. Usually, the maximum sampling rate 
/// is 240 Hz, but most channels are sampled at a frequency lower than that. 
/// Data stored in 'differential' mode are only stored when their value has 
/// changed from the previously stored value. This can save a tremendous 
/// amount of space especially for large but largely unchanged channels. 
/// Finally, variable length channels are array channels (i.e., cardinality > 1) 
/// that have only a subset of the array stored for each sample. This is done 
/// to save space, when the real-time software knows that only a subset of the 
/// array has valid data. 
///
class CDaqChannelInfo {
public:
	CDaqChannelInfo() {
		m_Id = m_Items = m_CapRate = m_Type = -1;
		m_VarLen = false;
		m_Name = "";
	}
	~CDaqChannelInfo() {};
    int				GetItemSize()	const;		///< A
	int				GetRecSize()	const;		///< B
	int				GetItemCount()	const;		///< C
	int				GetType()		const;		///< D
	int				GetId()			const;		///< E
	const string&	GetName()		const;		///< H
	int				GetCapRate()	const;		///< G
	bool			IsVarLen()		const;		///< H

	friend class CDaqLowLevelIo;

private:
	int		m_Id;
	int		m_Items;
	string	m_Name;
	int		m_CapRate;
	int		m_Type;
	bool	m_VarLen;
};


/// Return true, if channel is a variable length array
inline bool
CDaqChannelInfo::IsVarLen(void) const
{
	return m_VarLen;
}


///// Return sampling rate, in Hz
inline int
CDaqChannelInfo::GetCapRate(void) const
{
	return m_CapRate;
}

///// Return name as specified in cell file
inline const string &
CDaqChannelInfo::GetName(void) const
{
	return m_Name;
}

/// 
/// Return the channel identifier, a small integer that can be
/// used to reference the channel in other calls.
///
/// Keep in mind that the channel identifer can change among DAQ files,
/// even if it refers to the same cell variable.
///
inline int
CDaqChannelInfo::GetId(void) const
{
	return m_Id;
}

///// Return type (c=char, f=float, i=int, s=short, d=double
inline int
CDaqChannelInfo::GetType(void) const
{
	return m_Type;
}

///// Return number of items in each sample
inline int
CDaqChannelInfo::GetItemCount(void) const
{
	return m_Items;
}

/////////// Return record size, in bytes
inline int
CDaqChannelInfo::GetRecSize(void) const
{
	return GetItemSize() * GetItemCount(); 
}
