#ifndef _DAQIO_H_
#define _DAQIO_H_

/////////////////////////////////////////////////////////////////////////////
///\defgroup DAQ Data AcQuisition file
///\ingroup clib
///\ingroup DAQ 
///
///\class CDaqLowLevelIo
///
///\brief Low Level Reading from a DAQ
///
/// This class provides an API that does low level reading from a DAQ 
/// file. In particular, each DAQ file is treated as a set of independent 
/// channels, with each channel having its own type/cardinality/sampling 
/// rate and sampling method.
///
/// As a low level API, it only provides data integrity and raw data 
/// reading capability for either the whole daq file, or for selected 
/// channels. Except for these options, there are no other data conditioning 
/// options or partitioning options. 
///
class CDaqLowLevelIo {
public:
	CDaqLowLevelIo();
	~CDaqLowLevelIo();

	bool LowLevelDataRead(const TIntVec* , TDataVec& , std::vector<TIntVec>* pFrameList=0);
	bool LowLevelDataCopy(const TIntVec* chans, TFileVec& dest, 
				TConvertOptions options = eBINARY_FORMAT + eINCLUDE_FRAME);
	bool GetFullData(const TIntVec* chans, TDataVec& data, TConvertOptions option = eNO_OPTION,
				int firstFrame = -1, int lastFrame = -1, std::vector<TIntVec>* pFrameList = 0, int maxDiffLook = -1);

	bool           Open(const std::string &);				///< Open the file
	void           GetChannels(TChanInfoVec& channels) const;		///< Return info about channels
	bool           GetChannelInfo(int chId, CDaqChannelInfo &info) const;
	int            GetFirstFrame() const;
	int            GetLastFrame() const;  
	int            GetFrames() const;
	int            GetFrequency() const;
	const std::string&  GetDate() const;
	const std::string&  GetVersion() const;
	const std::string&  GetTitle() const;
	const std::string&  GetSubject() const;
	const std::string&  GetRun() const;
	const std::string&  GetRunInst() const;
	const std::string&  GetFileName() const;
	size_t         GetNumChannels() const;
	bool           TocFileExists() const;

	bool CheckIntegrity(bool buildToc = false, bool forceFileCreate = false);
	bool QueryIntegrityValues(TIntVec& drops, TIntVec& dropedFrm, TIntVec& skips, 
			TDaqEofStatus &stat);

	const std::string& GetLastError(void) const		///< Return text of most recent error
			{ return m_LastError; };
	void Close(void);							///< Close the file

	bool FrameDropped(int frm);
	bool FrameHasData(int frm);
	int  GetDroppedFrames() const;

	void SetCallbacks( void errf(const char *, const char *), void progf(const char *, int, int));


private:
	enum TReadErrorCode {
		eDAQ_READ_OK,			
		eDAQ_READ_ERROR,		///< read error (short file, premature eof etc.
		eDAQ_READ_EOF,			///< found end marker
		eDAQ_READ_FAILED,		///< lost sync
		eDAQ_READ_BAD_DATA,		///< bad data and couldn't recover
		eDAQ_NO_SUCH_FRAME,		///< asked to read a frame that doesn't exist in TOC
		eDAQ_READ_INTERNAL_ERROR
	};

	enum TResyncOutcome {
		eRESYNC_CHANNEL,	///< resynced ok, next thing to read is channel
		eRESYNC_FRAME,		///< resynced ok, next thing to read is frame
		eRESYNC_NONE		///< could not resync
	};

	bool ReadHeader(FILE *);
	bool ReadChannelInfo(FILE *);
	bool DetectFrameHeader(int recFrame);
	bool ResyncFrame(int curFrame, int& skipCount);
	TResyncOutcome ResyncChannel(int recFrame, int& skipCount);
	bool WriteTocFile(const std::string& fname);
	bool ReadTocFile(const std::string& fname);
	TReadErrorCode ReadFrameHeader(int,	int& ,int&, int&);
	TReadErrorCode ReadDataForOneFrame(int, TIntVec, TDataVec &, void *, std::vector<bool>&);
	TReadErrorCode ReadChannelDataForOneFrame(int, int, CDaqBuffer&, void*, bool&);

	std::string	m_Filename;
	std::string  m_Version;			///< file version
	FILE*	m_pFile;

	std::string	m_Title;			///< Simulation title, as stored in DAQ file
	std::string	m_Date;				///< Date of collection, as stored in DAQ file
	std::string	m_Subj;				///< As stored in DAQ file
	std::string	m_Run;				///< As stored in DAQ file
	std::string	m_RunInst;			///< As stored in DAQ file
	int		m_NumEntries;		///< As stored in DAQ file
	int		m_Frequency;		///< As stored in DAQ file

	void	(*m_UserErrorFunc)(const char *, const char *);
	void	(*m_UserProgrFunc)(const char *, int, int);

	bool                 m_TocExists;	///< Indicates a TOC exists in a file
	bool				 m_HaveToc;		///< Indicates if we have TOC in memory
	TChanInfoVec		 m_Channels;	///< Channel information
	long				 m_DataOffset;	///< where data starts in the file 
	std::map<int, int>        m_Toc;			///< given a sim frame, gives the offset in the file
	std::vector<int>	         m_DroppedFrames;
	int					 m_FirstFrame;
	int					 m_LastFrame;
	mutable std::string		 m_LastError;
	TDaqEofStatus        m_EofStatus;

	// These are for data integrity
	TIntVec				m_FrameDrops;		///< sequences of skipped frames
	TIntVec				m_Skips;			///< groups of bytes skipped
};


//////////////////////////////////////////////////////////////////////////////
///
/// Return an STL vector containing information about all the channels in the
/// DAQ file.
///
inline void
CDaqLowLevelIo::GetChannels(TChanInfoVec & chn) const
{
	chn = m_Channels;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the title stored in the DAQ file.
///
inline const std::string &
CDaqLowLevelIo::GetTitle(void) const
{
	return m_Title;
}


//////////////////////////////////////////////////////////////////////////////
///
/// Return the DAQ file version number as an STL string.

inline const std::string &
CDaqLowLevelIo::GetVersion(void) const
{
	return m_Version;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the run instance name.  If the feature is not
/// supported due to an older version, an empty string is returned.
///
inline const std::string &
CDaqLowLevelIo::GetRunInst(void) const
{
	return m_RunInst;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the subject name (also known as the participant).
///
inline const std::string &
CDaqLowLevelIo::GetSubject(void) const
{
	return m_Subj;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the name of the Run that produced the daq file
///
inline const std::string &
CDaqLowLevelIo::GetRun(void) const
{
	return m_Run;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the file creation date as stored in the header
///
inline const std::string &
CDaqLowLevelIo::GetDate(void) const
{
	return m_Date;
}


//////////////////////////////////////////////////////////////////////////////
///
/// Return the sampling frequency, in Hz.  The frequency is an integer.
///
inline int
CDaqLowLevelIo::GetFrequency(void) const
{
	return m_Frequency;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the total number of frames in the file
///
inline int
CDaqLowLevelIo::GetFrames(void) const
{
	return m_LastFrame - m_FirstFrame;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the first FBS frame number
///
inline int
CDaqLowLevelIo::GetFirstFrame(void) const
{
	return m_FirstFrame;
}

//////////////////////////////////////////////////////////////////////////////
///
/// Return the last FBS frame number
///
inline int
CDaqLowLevelIo::GetLastFrame(void) const
{
	return m_LastFrame;
}

//////////////////////////////////////////////////////////////////////////////
///
/// This funcion returns the number of channels in the DAQ file
///
inline size_t
CDaqLowLevelIo::GetNumChannels(void) const
{
	return m_Channels.size();
}

//////////////////////////////////////////////////////////////////////////////
///
/// This function returns the name of the DAQ file
///
inline const std::string&  
CDaqLowLevelIo::GetFileName() const 
{ 
	return m_Filename; 
};

//////////////////////////////////////////////////////////////////////////////
///
/// Returns a boolean indicating if the toc already exists.
///
inline bool           
CDaqLowLevelIo::TocFileExists() const 
{ 
	return m_TocExists; 
};

#endif // _DAQIO_H_