#ifndef _DAQBUFFER_H_
#define _DAQBUFFER_H_

//////////////////////////////////////////////////////////////////////////////////////////
///
/// $Id: DaqBuffer.h,v 1.6 2020/03/12 00:43:37 IOWA\dheitbri Exp $
/// 
/// Date: December 2004
///
/// Description: 
/// A small class that maintains a buffer 
///

/////////////////////////////////////////////////////////////////////////////////////////
///
/// \ingroup clib
/// \ingroup DAQ
///
/// This support class provides buffer space for copying daq data out of a 
/// DAQ file. It is used in lieu of raw buffers to minimize the likelihood of 
/// memory leaks.
///
/// The class itself is not aware of the type or cardinality of the data, 
/// however, it provides a few type-specific access functions to facilitate 
/// data access when the user knows the type of data.
///
class CDaqBuffer {
public:
	CDaqBuffer() { m_Count = 0; };
	~CDaqBuffer() {};


	std::vector<TDaqByte>::const_pointer GetDataPtr() const;	///< Return faw pointer to data

    size_t  GetCount() const;		///< Return number of typed elements in buffer
    size_t  GetSize() const;		///< Return buffer size in bytes
	char*   GetChar(int ind=0);		///< Return char type pointer to data
	int*    GetInt(int ind=0);      ///< Return int type pointer to data
	float*  GetFloat(int ind=0);	///< Return float type pointer to data
	double* GetDouble(int ind=0);	///< Return double type pointer to data
	short*  GetShort(int ind=0);	///< Return short type pointer to data

	friend class CDaqLowLevelIo;
	std::vector<TDaqByte>    m_Data;     ///< data is stored here

protected:
	void Append(const void *pData, int size, int count);		///< Add data
	void Replace(const void *pData, size_t size, int count);	///< Replace data
	void AllocSpace(int size, int count=0);						///< Allocate space
	void Clear(void);											///< Release everything
	int                 m_Count;								///< count of data, not in bytes but in type
};


///
/// This function ...
//
inline size_t             
CDaqBuffer::GetSize() const
{ 
	return (int)m_Data.size(); 
};

///
/// This function ...
inline size_t
CDaqBuffer::GetCount() const
{ 
	return m_Count; 
}

///
/// Return ...
///
inline char*
CDaqBuffer::GetChar(int ind)
{ 
	return ((char *)&m_Data[0]) + ind; 
};

///
/// Return ...
///
inline int*    
CDaqBuffer::GetInt(int ind)
{ 
	return ((int *)&m_Data[0]) + ind; 
};

///
/// This function ...
//
inline float*  
CDaqBuffer::GetFloat(int ind)     ///< Return float type pointer to data
{ 
	return ((float *)&m_Data[0]) + ind; 
};

///
/// This function ...
//
inline double* 
CDaqBuffer::GetDouble(int ind)    ///< Return double type pointer to data
{ 
	return ((double *)&m_Data[0]) + ind; 
};

///
/// This function ...
//
inline short*  
CDaqBuffer::GetShort(int ind)     ///< Return short type pointer to data
{ 
	return ((short *)&m_Data[0]) + ind; 
};


////////////////////////////////////////////////////////////////////////////////
///
/// This function returns a raw pointer to the data in the buffer.
/// 
inline std::vector<TDaqByte>::const_pointer 
CDaqBuffer::GetDataPtr() const 
{ 
	if (m_Data.size() > 0)
		return &m_Data[0];
	else 
		return NULL;
};


/////////////////////////////////////////////////////////////////////////////////
///
/// This function adds data to the DaqBuffer.  It is used by the DAQ library
/// to build the buffers returned to the user, and as such, is not meant to
/// be used by the library client.
///
/// The function receives a pointer to the data, the size of the buffer in 
/// bytes and a count of elements in that buffer.  The class deals with
/// untyped buffers, so the only way to keep track of how many elements
/// are in the buffer (i.e., 2 integers as opposed to 8 bytes) is for the
/// caller to provide that information, which in this case is the count 
/// argument.
///
inline void
CDaqBuffer::Append(
	const void*  p,      ///< pointer to data to append
	int		     size,   ///< size in bytes of data
	int          count)  ///< count of data items in that buffer
{
	size_t oldsize  = m_Data.size();

	m_Data.resize(oldsize + size);
	memcpy(&m_Data[0]+oldsize, p, size);
	
	m_Count += count;
}

/////////////////////////////////////////////////////////////////////////////////
///
/// This function replaces the existing data with new one. 
///
/// The function receives a pointer to the data, the size of the buffer in 
/// bytes and a count of elements in that buffer.  The class deals with
/// untyped buffers, so the only way to keep track of how many elements
/// are in the buffer (i.e., 2 integers as opposed to 8 bytes) is for the
/// caller to provide that information, which in this case is the count 
/// argument.
///
inline void
CDaqBuffer::Replace(
	const void*  p,      ///< pointer to data to append
	size_t	     size,   ///< size in bytes of data
	int          count)  ///< count of data items in that buffer
{
	size_t oldsize  = m_Data.size();

	if ( oldsize < size ) {
		m_Data.resize(size);
	}
	memcpy(&m_Data[0], p, size);
	
	m_Count = count;
}


/////////////////////////////////////////////////////////////////////////////////
///
/// This function allocates space in the DaqBuffer.  
///
///
inline void
CDaqBuffer::AllocSpace(
	int     size,   ///< size in bytes of data
	int     count)  ///< count of data items in that buffer
{
	m_Data.resize(size);
	m_Count = count;
}


/////////////////////////////////////////////////////////////////////////////////
///
/// This function allocates space in the DaqBuffer.  
///
inline void 
CDaqBuffer::Clear(void) 
{ 
	m_Data.clear(); m_Count = 0; 
}

#endif  // _DAQBUFFER_H_