/////////////////////////////////////////////////////////////////////////////
///
// DAQ low level IO Library
// 
// $Id: DaqFileIO.cxx,v 1.27 2020/03/12 00:43:37 IOWA\dheitbri Exp $
// Date: March, 2005
// Author: Yiannis Papelis
//
/// \defgroup clib C++ Library 
///
/// @{
///
#include "DaqIoLib.h"
#include <assert.h>
#include <io.h>

using namespace std;
#ifdef WIN32
    #pragma warning(push)
    #pragma warning(disable: 4996)
#endif
///
/// Small dummy function to act as the default callback
///
static void DefaultErrorFunc(const char *, const char *) {}

///
/// Default functionality for the progress callback
///
static void DefaultProgressFunc(const char *, int, int) {}

struct TFrameInfo {
	long long ofs;
	int numChan;
};
//Intel says 8k (2006) is the ideal buffer size for streaming from the HD,
//this may need to be changed with time.
const int cFILE_BUFFER_SIZE = 8192;

CDaqLowLevelIo::CDaqLowLevelIo()
{
	m_pFile         = 0;
	m_UserErrorFunc = DefaultErrorFunc;
	m_UserProgrFunc = DefaultProgressFunc;

	m_HaveToc    = false;
	m_FirstFrame = -1;
	m_LastFrame  = -1;
	m_EofStatus  = eDAQ_EOF_UNDEFINED;
	m_TocExists  = false;

}

CDaqLowLevelIo::~CDaqLowLevelIo()
{
	if ( m_pFile )
		Close();
}



int
CDaqChannelInfo::GetItemSize() const
{
	if ( m_Type == 'f' || m_Type == 'i' ) return 4;
	if ( m_Type == 'd' ) return 8;
	if ( m_Type == 's' ) return 2;
	if ( m_Type == 'c' ) return 1;
	assert(0);
	return -1;
}


//////////////////////////////////////////////////////////////////////////////////////
///
/// This function allows user to specify error and progress callbacks. The 
/// error callback provides a string explaining the error and the progress 
/// callback provides actual/total frames processed. The exact meaning of the 
/// progress function arguments depends on what activity the library is performing 
/// when it calls it. 
///
void
CDaqLowLevelIo::SetCallbacks(
	void errf(const char *, const char *),
	void progf(const char *, int, int)
	)
{
	if ( errf != 0 ) m_UserErrorFunc = errf; else m_UserErrorFunc = DefaultErrorFunc;
	if ( progf != 0 ) m_UserProgrFunc = progf; else m_UserProgrFunc = DefaultProgressFunc;
}


///
/// Doc
///
bool
CDaqLowLevelIo::GetChannelInfo(
	int              chId, 
	CDaqChannelInfo& info) const
{
	if ( m_pFile == 0 ) {
		m_LastError = "uninitialized class instance (call Open first)";
		return false;
	}

	if ( chId < 0 || chId >= (int)m_Channels.size() ) {
		m_LastError = "invalid channel";
		return false;
	}
	info = m_Channels[chId];

	m_LastError = "";
	return true;
}

///
/// adf
///
void
NearDump(FILE* p, int ofs)
{
	auto origOfs = _ftelli64(p);

	printf("\nData dump near unrecoverable read at offset %d\n", ofs);
    _fseeki64(p, -8*4, SEEK_CUR);
	int d;

	for (int i=0; i<96; i++) {
		size_t n = fread(&d, sizeof(d), 1, p);
		if ( n == 1 ) {
			printf("%-d:  0x%08X %d\n", int(ftell(p)-sizeof(d)), d, d);
		}
		else {
			printf("Read at ofs %-d failed.\n", int(ftell(p)-sizeof(d)));
		}
	}
    _fseeki64(p, origOfs, SEEK_SET);
}

 
/////////////////////////////////////////////////////////////////////////////
///
/// Utility function that writes ASCII data given an untyped pointer to it.  
///
///
static bool 
WriteAsciiLine(
	FILE*             pFile, 
	CDaqChannelInfo&  chInf, 
	const void*       pData, 
	TConvertOptions   options)
{
	assert(pFile);

	int sample;

	if ( chInf.GetType() == 'c' ) {
		for (sample=0; sample < chInf.GetItemCount(); sample++) {
			// skip strings for now
		}
	}
	else if ( chInf.GetType() == 'f' ) {
		float *p = (float *)pData;
		for (sample=0; sample < (int)chInf.GetItemCount(); sample++) {
			fprintf(pFile, "%E ", *p);
			p++;
		}
	}
	else if ( chInf.GetType() == 'i' ) {
		int *p = (int *)pData;
		for (sample=0; sample < (int)chInf.GetItemCount(); sample++) {
			fprintf(pFile, "%d ", *p);
			p++;
		}
	}
	else if ( chInf.GetType() == 'd' ) {
		double *p = (double *)pData;
		for (sample=0; sample < (int)chInf.GetItemCount(); sample++) {
			fprintf(pFile, "%E ", *p);
			p++;
		}
	}
	else if ( chInf.GetType() == 's' ) {
		int *p = (int *)pData;
		for (sample=0; sample < (int)chInf.GetItemCount(); sample++) {
			fprintf(pFile, "%d ", *p);
			p++;
		}
	}
	else {
		assert(0);
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
///
/// This function determines if a frame was dropped by the DAQ.  The
/// function requires that integrity testing has taken place (i.e., a TOC
/// exists.
///
bool
CDaqLowLevelIo::FrameDropped(int frm)
{
	vector<int>::const_iterator    pDrpFrm;	// iterates list of dropped frames
	for (pDrpFrm = m_DroppedFrames.begin(); pDrpFrm != m_DroppedFrames.end(); pDrpFrm++){
		if ( frm == *pDrpFrm )return true;
	}
	return false;
}

int  CDaqLowLevelIo::GetDroppedFrames() const {
	return (int)m_DroppedFrames.size();
}
/////////////////////////////////////////////////////////////////////////////
///
/// This function determines if data was logged for a given frame
///
bool
CDaqLowLevelIo::FrameHasData(int frm)
{
	map<int, long long>::const_iterator p;

	p = m_Toc.find(frm);
	return p != m_Toc.end();
}

/////////////////////////////////////////////////////////////////////////////
///
/// Reads the header of the DAQ file, independent of its version, and 
/// stores the information in variables local to the class instance.  
/// Values that are not provided in the specific file version are set
/// to some default
///
bool
CDaqLowLevelIo::ReadHeader(	FILE *p)
{
	unsigned magic;

	if ( fread(&magic, sizeof(magic), 1, p) != 1 ) {
		m_LastError = "cannot read magic number";
		return false;
	}
	rewind(p);

	bool supported = false;

	if  ( magic == DAQ_MAGIC_NUM_VER_2_0 ) {
		m_Version = "2.0";
		struct Hdr {
			unsigned Magic;
			char     Title[128];
			char     Date[27];
			char     Subj[64];
			char     Run[64];
			int      NumEntries;
			unsigned DAQFrequency;
		} header = { 0 };

		if ( fread(&header, sizeof(header), 1, p) != 1 ) {
			m_LastError = "cannot read header";
			return false;
		}
		m_Title      = header.Title;
		m_Date       = header.Date;
		m_Subj       = header.Subj;
		m_Run        = header.Run;
		m_RunInst    = "Unavailable";
		m_NumEntries = header.NumEntries;
		m_Frequency  = header.DAQFrequency;
		supported = true;
	}
	else if ( magic == DAQ_MAGIC_NUM_VER_2_1 ) {
		printf("Version 2.1\n");
		exit(0);
	}
	else if ( magic == DAQ_MAGIC_NUM_VER_2_2 ) {
		m_Version = "2.2";

		struct Hdr {
			unsigned Magic;
			char     Title[120];
			char     Date[27];
			char     Subj[128];
			char     Run[128];
			char     RunInst[128];
			int      NumEntries;
			unsigned DAQFrequency;
		} header = { 0 };

		if ( fread(&header, sizeof(header), 1, p) != 1 ) {
			m_LastError = "cannot read header";
			return false;
		}
		m_Title      = header.Title;
		m_Date       = header.Date;
		m_Subj       = header.Subj;
		m_Run        = header.Run;
		m_RunInst    = header.RunInst;
		m_NumEntries = header.NumEntries;
		m_Frequency  = header.DAQFrequency;
		supported = true;
	}

	if ( !supported ) {
		fprintf(stderr, "Unsuppored version, magic=0x%X\n", magic);
		return false;
	}

	m_LastError = "";
	return true;
}


/////////////////////////////////////////////////////////////////////////////
///
/// Reads the information about all channels stored in the DAQ file.
/// Dagta is stored in the m_Channels instance variable
///
bool
CDaqLowLevelIo::ReadChannelInfo(FILE *p)
{
	int     ch;

    struct ChInfo {
      int   ItemCount;
      char  Name[36];
      char  Units[16];
      short CaptureRate;
      int   DataType;
      char  VarLenFlag;
    } chan;


	for (ch=0; ch < m_NumEntries; ch++) {
		CDaqChannelInfo  chInfo;

		if ( fread(&chan, sizeof(chan), 1, p) != 1 ) {
			return false;
		}

		chInfo.m_Id      = ch;
		chInfo.m_Items   = chan.ItemCount;
		chInfo.m_Name    = chan.Name;
		chInfo.m_Type    = chan.DataType;
		chInfo.m_CapRate = chan.CaptureRate;
		chInfo.m_VarLen  = chan.VarLenFlag ? true : false;

		m_Channels.push_back(chInfo);
	}

	return true;
}


//////////////////////////////////////////////////////////////////////////////////
///
/// Returns true if the file is currently seeked on a frame header else false.
/// Either way, it leaves the file pointer at the same place it found it unless
/// it run into a read error.
///
bool
CDaqLowLevelIo::DetectFrameHeader(
	int recentFrame)
{
	auto origOfs = _ftelli64(m_pFile);
	int header[3];

	if ( fread(header, sizeof(int), 3, m_pFile) != 3 ) return false;

	// we use a large number for the reasonable frame skip since with
	// small records, a single DAQ buffer can often be as many frames
	bool reasonableFrame   = header[1] - recentFrame < 200000;
	bool reasonableChCount = header[2] >= 0 && header[2] < m_NumEntries;

	_fseeki64(m_pFile, origOfs, SEEK_SET);

	if ( header[0] == -1 && reasonableFrame && reasonableChCount ) {
		return true;
	}
	else {
		return false;
	}
}


//////////////////////////////////////////////////////////////////////////////////
///
/// Attempts to re-sync after reading a bad channel definition;  usually, 
/// there are more channel definitions nearby so we can re-sync there
///
CDaqLowLevelIo::TResyncOutcome
CDaqLowLevelIo::ResyncChannel(
	int		recentFrame,
	long long&	skip)
{
	int  data;
	int  cch1;		// candidate channels
	int  tries = 0;
	auto  startingOffset = _ftelli64(m_pFile);

	skip = 0;
	while ( !feof(m_pFile) && tries++ < cMaxChannResyncLookahead) {
		long long mostRecentGoodOfs;

		mostRecentGoodOfs = _ftelli64(m_pFile);
		if ( fread(&data, sizeof(int), 1, m_pFile) != 1 ) return eRESYNC_NONE;

		// possible channel definition; read at least two more things to be sure
		// we are back in sync
		if ( data >=0 && data < m_NumEntries ) {
			cch1 = data;
			CDaqChannelInfo& ch = m_Channels[cch1];
			long long  skipDataPortion = ch.GetRecSize();

            _fseeki64(m_pFile, skipDataPortion, SEEK_CUR);

			if ( fread(&data, sizeof(int), 1, m_pFile) != 1 ) return eRESYNC_NONE;

			if ( data >=0 && data < m_NumEntries && data > cch1 ) {
				// it seems as if we are back in sync, we re-wind and return
				while ( data >=0 && data < m_NumEntries && data > cch1 ) {
					CDaqChannelInfo& ch = m_Channels[data];
					long long  skipDataPortion = ch.GetRecSize();

                    _fseeki64(m_pFile, skipDataPortion, SEEK_CUR);
					if ( fread(&data, sizeof(int), 1, m_pFile) != 1 ) return eRESYNC_NONE;
				}
			}

			if ( data == -1 ) {
				fseek(m_pFile, -4, SEEK_CUR);
				if ( DetectFrameHeader(recentFrame)) {
					skip = _ftelli64(m_pFile) - startingOffset;
					return eRESYNC_FRAME;
				}
				fseek(m_pFile, 4, SEEK_CUR);
			}
		}
		else if ( data == -1 ) {
			fseek(m_pFile, -4, SEEK_CUR);
			if ( DetectFrameHeader(recentFrame)) {
				skip = _ftelli64(m_pFile) - startingOffset;
				return eRESYNC_FRAME;
			}
			fseek(m_pFile, 4, SEEK_CUR);
		}
		else {
			fseek(m_pFile, -3, SEEK_CUR);
			skip++;
		}
	}

	return eRESYNC_NONE;
}


bool
CDaqLowLevelIo::ResyncFrame(
	int   curFrame,     // last known frame, if -1 then we have no info
	long long&  bytesSkipped)
{
	int head[3];

	bytesSkipped = 0;
	while ( !feof(m_pFile) ) {
		if ( fread(head, sizeof(int), 3, m_pFile) != 3 ) return false;

		//
		// Look for a record header
		//
		if ( head[0] == -1 ) {
			bool reasonableFrame = curFrame == -1 || head[1] - curFrame < 20000;
			bool reasonableCount = head[2] >=0 && head[2] < m_NumEntries;

			if ( reasonableFrame && reasonableCount ) {
                _fseeki64(m_pFile, -3 * (signed int)sizeof(int), SEEK_CUR);
				return true;
			}
		}
		//
		// Look for an end record
		//
		else if ( head[0] == -2 ) {
			return true;            // NEEDS MORE WORK
		}

		//
		// Since we've lost allignment, we can't keep reading, we have to
		// keep looking for a frame header one byte at a time
		else {
			_fseeki64(m_pFile, 1 - 3 * sizeof(int), SEEK_CUR);
			bytesSkipped++;
		}
	}
	return false;
}


void
CDaqLowLevelIo::Close(void)
{
	if (m_pFile)
		fclose(m_pFile);
	m_pFile         = 0;


	m_UserErrorFunc = DefaultErrorFunc;
	m_UserProgrFunc = DefaultProgressFunc;

	m_HaveToc    = false;
	m_FirstFrame = -1;
	m_LastFrame  = -1;
	m_EofStatus  = eDAQ_EOF_UNDEFINED;
	m_TocExists  = false;
}


bool
CDaqLowLevelIo::Open(
	const string &  fname)
{
	//
	// Check if we have already opened the file
	//

	if ( m_pFile ) {
		m_LastError = "file already open";
		return false;
	}

	// 
	// Open the file
	// 
	m_pFile = fopen(fname.c_str(), "rb");
	if ( m_pFile == 0 ) {
		m_LastError = "cannot open file '";
		m_LastError += fname;
		m_LastError += "': ";
		m_LastError += strerror(errno);
		return false;
	}

	m_Filename = fname;

	//
	// Read the header
	//
	if ( !ReadHeader(m_pFile) ) {
		fclose(m_pFile);
		m_pFile = 0;
		return false;
	}

	//
	// Read channel information 
	//
	if ( !ReadChannelInfo(m_pFile) ) {
		fclose(m_pFile);
		m_pFile = 0;
		return false;
	}

	m_DataOffset = _ftelli64(m_pFile);

	// find the first frame
	int  maxTries = 2000;
	int  recHead[3];    // (Code, Frame, Count)
	
	while ( maxTries-- ) {
		if ( fread(recHead, sizeof(int), 3, m_pFile) != 3 ) break;

		if ( recHead[0] == -1 ) {
			m_FirstFrame = recHead[1];
			break;
		}
	}

	// try to find out the last frame so we have an idea of the total
	// number of frames in the file
	fseek(m_pFile, -12, SEEK_END);
	m_LastFrame = -1;
	maxTries = 64*1024;
	while ( maxTries-- ) {
		if ( fread(recHead, sizeof(int), 3, m_pFile) != 3 ) break;

		if ( recHead[0] == -2 && recHead[2] == DAQ_END_MARK ) { // score !!
			m_EofStatus = eDAQ_EOF_OK;
			m_LastFrame = recHead[1];
			break;
		}
		else if ( recHead[0] == -1 && recHead[2] >=0 && recHead[2] < m_NumEntries ) {
			m_EofStatus = eDAQ_EOF_PARTIAL_MARK;
			m_LastFrame = recHead[1];
			break;
		}
		else {
			fseek(m_pFile, -( 3* (signed)sizeof(int) + 1), SEEK_CUR);
		}
	}

	if ( m_LastFrame == -1 ) {
		m_EofStatus = eDAQ_EOF_NOTFOUND;
	}

	//
	// No matter what happened with the last frame search, we seek ready to
	// read data
	//
	_fseeki64(m_pFile, m_DataOffset, SEEK_SET);

	//
	// Now look for the TOC file, if one exists read it
	//
	if ( ReadTocFile(fname + ".toc") )
		m_TocExists = true;

	m_LastError = "";
	return true;
}

//////////////////////////////////////////////////////////////////////////////////
///
/// This function writes the TOC into a file where it can later be retrieved
/// without having to go through the DAQ file.
///
/// The file contains:
///		magic (int)
///		offset of table (int) (i.e., where first frame, offset pair stored
///		first_frame (int)
///		last_frame (int)
///		num_entries in toc (int)
///		num_framedrops (int)
///			frame drop 1 len (int)
///			frame drop 2 len (int)
///			...
///     num_droppedframes (int)
///			dropped frame 1 (int)
///			dropped frame 2 (int)
///			...
///		num_skips (int)
///			skip 1 len (int)
///			skip 2 len (int)
///			...
///		frame, offset (int, int)
///		frame, offset (int, int)
///		...
///
bool
CDaqLowLevelIo::WriteTocFile(
	const string& fname)
{
	FILE *p = fopen(fname.c_str(), "wb");
	if ( p == 0 ) return false;

	int d1;
	long long tblOfs, d2;

	d1 = DAQ_TOC_MAGIC_NUM64;
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// again, leave space for table offset
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// first frame
	d1 = m_FirstFrame;
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// last frame
	d1 = m_LastFrame;
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// num entries
	d1 = (int)m_Toc.size();
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	TIntVec::iterator vi;
	map<int,long long>::iterator mi;

	// num frame drops
	d1 = (int)m_FrameDrops.size();
   	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// frame drops
	for (vi = m_FrameDrops.begin(); vi != m_FrameDrops.end(); vi++) {
		d1 = *vi;
		if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;
	}

	// num dropped frames
	d1 = (int)m_DroppedFrames.size();
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// dropped frames
	for (vi = m_DroppedFrames.begin(); vi != m_DroppedFrames.end(); vi++) {
		d1 = *vi;
		if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;
	}

	// num skips
	d1 = (int)m_Skips.size();
	if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;

	// skips
	for (vi = m_Skips.begin(); vi != m_Skips.end(); vi++) {
		d1 = *vi;
		if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;
	}

	tblOfs = _ftelli64(p);

	for (mi = m_Toc.begin(); mi != m_Toc.end(); mi++) {
		d1 = mi->first;
		d2 = mi->second;
		if ( fwrite(&d1, sizeof(int), 1, p) != 1 ) return false;
		if ( fwrite(&d2, sizeof(long long), 1, p) != 1 ) return false;
	}

	fseek(p, 4, SEEK_SET);
	if ( fwrite(&tblOfs, sizeof(int), 1, p) != 1 ) return false;

	fclose(p);
	return true;

}


//////////////////////////////////////////////////////////////////////////////////
///
/// This function reads the TOC from a file.  If anything goes wrong with
/// reading the file, the function quietly returns and doesn't change anything.
///
/// The function returns true to indicate it found and successfully loaded a 
/// toc file, else it returns false
///
bool
CDaqLowLevelIo::ReadTocFile(
	const string& fname)
{
	TIntVec			FrameDrops;
	TIntVec			DroppedFrames;
	TIntVec			Skips;
	int				numEntries;
	int				i, count, d1;
    long long       d2;

	//
	// Cannot read the TOC for a file which we have not yet opened
	//
	if ( m_pFile == 0 ) {
		return false;
	}

	FILE *p = fopen(fname.c_str(), "rb");
	if ( p == 0 ) {
		return false;
	}

	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	if ( d1 != DAQ_TOC_MAGIC_NUM && d1 != DAQ_TOC_MAGIC_NUM64) return false;
    
    size_t refSize = 4;
    if (d1 == DAQ_TOC_MAGIC_NUM64) {
        refSize = 8;
    }
	// we don't need the offset now, it's for future compatibility
	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;

	// Let the TOC contents override auto-calc for first/last frame
	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	m_FirstFrame = d1;

	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	m_LastFrame = d1;

	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	numEntries = d1;

	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	count = d1;
	for (i=0; i<count; i++) {
		if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
		FrameDrops.push_back(d1);
	}

	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	count = d1;
	for (i=0; i<count; i++) {
		if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
		DroppedFrames.push_back(d1);
	}

	if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
	count = d1;
	for (i=0; i<count; i++) {
		if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
		Skips.push_back(d1);
	}
	map<int, int>	Toc;

	vector<int>		frame;
	vector<long long> offset;
	for (i=0; i<numEntries; i++) {
		if ( fread(&d1, sizeof(int), 1, p) != 1 ) return false;
        if (refSize == 4) {
            int tempVal;
            if (fread(&tempVal, refSize, 1, p) != 1) return false;
            d2 = tempVal;
        }
        else {
            if (fread(&d2, refSize, 1, p) != 1) return false;
        }
		frame.push_back(d1);
		offset.push_back(d2);
	}
	fclose(p);
	assert(frame.size() == numEntries);
	assert(offset.size() == numEntries);

	// all is ok, so copy data
	m_Toc.clear();
	for (i=0; i<numEntries; i++) {
		m_Toc[frame[i]] = offset[i];
	}

	m_FrameDrops    = FrameDrops;
	m_DroppedFrames = DroppedFrames;
	m_Skips         = Skips;

	m_HaveToc = true;
	return true;
}


//////////////////////////////////////////////////////////////////////////////////
///
/// This function returns information about the file's integrity. It can 
/// only be called after CheckIntegrity has successfully returned.
/// 
/// The function returns (in the provided arguments) two arrays of 
/// integers. The first array has an element for each time time frame drops 
/// were encountered. The actual array element is the number of frames that 
/// were dropped. Similarly, the second array has the number of times data 
/// has to be skipped to re-sync; one element is used for each successful 
/// re-sync attempt, the actual element value is the number of bytes that 
/// had to be skipped. 
///
bool 
CDaqLowLevelIo::QueryIntegrityValues(
	TIntVec& drops, 
	TIntVec& dropedFrm,
	TIntVec& skips,
	TDaqEofStatus& eofStat)
{
	if ( m_HaveToc == false ) return false;

	drops     = m_FrameDrops;
	dropedFrm = m_DroppedFrames;
	skips     = m_Skips;
	eofStat   = m_EofStatus;

	return true;
}


//////////////////////////////////////////////////////////////////////////////////
///
/// This function verifies the integrity of the DAQ file by going through it 
/// and checking all records for consistency. In addition, the function builds 
/// a table of contents that allows better random access to the contents 
/// of the file. Once this function has returned successfully, the functions 
/// that allow access to the subset of the frames can be used.
///
/// By default, the table of contents is kept in memory, however, if the
/// buildTocFile argument is true, then the function will write the TOC into
/// a separate file that has the same name as the DAQ file but with the
/// extension .toc added after the name.  The TOC file is a binary
/// representation of the toc and integrity values.  Note that no check
/// is made for prior existence of the TOC file.  
///
/// By default, any errors that occur while tring to create the TOC file
/// are silently ignored.  If the forceFileCreate flag is set, then any
/// such error is treated as a fatal error and the function will return
/// an error code.
///
/// The function returns true to indicate the file is ok (or has problems 
/// that can be fixed) or false to indicate an unrecoverable error. 
/// To determine the existence of dropped frames and any other recoverable 
/// error, use the QueryIntegrityValues() function.
///
/// The function will call, within its context, the progress callback as 
/// well as the error callback. Note that the error callback is only called 
/// for recoverable errors which allow  the function to continue working.
/// Unrecoverable errors that cause the function to return false do not 
/// trigger the error callback.
///
bool
CDaqLowLevelIo::CheckIntegrity(
	bool	writeTocFile,
	bool	forceFileCreate
)
{
	char	buf[256];
	int		oldFrame   = -1;
	int		numFrames  = 0;
	map<int, TFrameInfo>  tempToc;
	bool    done = false;
	int     largestFrame = -1;

	//
	// Go through the file and build the TOC.  Only frames with data that
	// have shown no data skips or other problems are kept
	// 
	m_FirstFrame = -1;
	while ( !done ) {
		int  curFrame = -1;     // frame for current record
		int  numChan;           // number of channels in current record
		long long  skip = 0;
		long long  frameOffset;

		CDaqLowLevelIo::TReadErrorCode  rcode;

		rcode = ReadFrameHeader(-1, curFrame, numChan, skip);
		if ( rcode == eDAQ_READ_OK || rcode == eDAQ_READ_EOF ) {
			if ( m_FirstFrame == -1 ) m_FirstFrame = curFrame;

			if ( (numFrames % 720) == 0 ) (*m_UserProgrFunc)(m_Filename.c_str(), 
						numFrames, m_LastFrame - m_FirstFrame);

			if ( curFrame > largestFrame ) largestFrame = curFrame;

			frameOffset = _ftelli64(m_pFile) - 12;

			numFrames++;
			if ( rcode == eDAQ_READ_EOF ) {
				int tot = m_LastFrame - m_FirstFrame;
				(*m_UserProgrFunc)(m_Filename.c_str(), tot, tot);
				break;
			}
		}
		else if ( rcode == eDAQ_READ_ERROR ) {	// bad code read, couldn't resync
			sprintf(buf, "Could not recover after error at file offset %d", ftell(m_pFile));
			m_LastError = buf;
			return false;
		}
		else if ( rcode == eDAQ_READ_FAILED ) {	// fread  failed
			sprintf(buf, "Could not read data near offset %d", ftell(m_pFile));
			m_LastError = buf;
			return false;
		}
		else {
			m_LastError =  "General read error";
			return false;
		}

		// print warning and keep track of frame skips
		if ( skip > 0 ) {
			m_Skips.push_back(skip);
			sprintf(buf, "Data error before %d, resync'ed successfuly", ftell(m_pFile));
			(*m_UserErrorFunc)(m_Filename.c_str(), buf);
		}

		// Now read information about data in this record and skip over data portion
		int ch;
		bool problems = false;

		ch = 0;
		while ( ch < numChan ) {
			int chId;
			long long ofs= _ftelli64(m_pFile);		// to keep track of where errors may occur

			if ( fread(&chId, sizeof(int), 1, m_pFile) != 1 ) {
				if ( m_EofStatus == eDAQ_EOF_NOTFOUND ) { // OK, we had no end of file
					m_LastFrame = curFrame;
					done        = true;
					(*m_UserErrorFunc)(m_Filename.c_str(), "reached EOF with no end-marker");
					break;
				}
				else if ( m_EofStatus == eDAQ_EOF_PARTIAL_MARK && curFrame == m_LastFrame ) {
					m_LastFrame = curFrame;
					done        = true;
					break;
				}
				else {
					sprintf(buf, "Couldn't read file, premature end");
					m_LastError = buf;
					return false;
				}
			}
			if ( chId >= m_NumEntries || chId < 0 ) {
				// NearDump(m_pFile, ofs);
				TResyncOutcome rcode;
				int errOfs = ftell(m_pFile);

				problems = true;
				rcode = ResyncChannel(curFrame, skip);
				if ( rcode == eRESYNC_CHANNEL || rcode == eRESYNC_FRAME ) {
					sprintf(buf, "Invalid channel %d @ ofs %d, resync'ed ok", chId, errOfs);
					(*m_UserErrorFunc)(m_Filename.c_str(), buf);
					break;
				}
				else {
					sprintf(buf, "Invalid channel %d @ ofs %d, could not resync", chId, errOfs);
					m_LastError = buf;
					return false;
				}
			}
			else {
				CDaqChannelInfo& ch = m_Channels[chId];
				int              skip;

				if ( ch.IsVarLen() ) {
					int actCount;

					if ( fread(&actCount, sizeof(int), 1, m_pFile) != 1 ) {
						sprintf(buf, "Could not read var size channel length; chId=%d, ofs=%I64d, abort",
							chId, _ftelli64(m_pFile));
						m_LastError = buf;
						(*m_UserErrorFunc)(m_Filename.c_str(), buf);
						return false;
					}
					skip = ch.GetItemSize() * actCount;
				}
				else {
					skip = ch.GetRecSize();				
				}
				if ( skip ) fseek(m_pFile, skip, SEEK_CUR);			// save a system call if 0
			}
			ch++;

			// This code addresses a specific bug in the DAQ that would write
			// the wrong number of channels in the frame header.  What happens in
			// such cases is taht the next integer, instead of -1 (as the frame 
			// header) would be a channel number containing valid data.  
			if ( ch == numChan ) {
				int  oldChId = chId;
				int  numRead = (int)fread(&chId, sizeof(int), 1, m_pFile);
				bool okChan  = chId > oldChId && chId < m_NumEntries;

				if ( numRead == 1 && okChan ) {
					sprintf(buf, "Found extra channel %d in frame %d", chId, curFrame);
					(*m_UserErrorFunc)(m_Filename.c_str(), buf);
					ch--;
					fseek(m_pFile, -4, SEEK_CUR);
					continue;
				}
				else {
					fseek(m_pFile, -4, SEEK_CUR);
				}
			}
		}

		if ( !problems ) { // we only want frames for which data was intact
			TFrameInfo info;

			info.numChan      = numChan;
			info.ofs          = frameOffset;
			tempToc[curFrame] = info;
		}
	}

	// 
	// At this point, we have the temporary TOC file which we can use
	// to calculate actual frame drops.  AFter this is done, we can copy
	// into the final TOC only frames that carry data
	//
	{
		int frm = m_FirstFrame;
		int old = -1;
		int curFrame;

		// We have to be careful in messed up files because the m_LastFrame calculated
		// by the Open... is the last frame found in the file; In messed up files, the
		// last frame found is not necessarily the last frame with data
		if ( largestFrame > m_LastFrame ) {
			sprintf(buf, "Highest number frame (%d) wasn't last one in file,"
				"which was %d", largestFrame, m_LastFrame);
			(*m_UserErrorFunc)(m_Filename.c_str(), buf);
			m_LastFrame = largestFrame;
		}
		while ( frm < m_LastFrame ) {
			if ( tempToc.find(frm) != tempToc.end() ) {// if it's in the map, it was in the file 
				curFrame = frm;
				if ( old != -1 && curFrame - old > 1 ) {
					int len = (curFrame-old) - 1;
					m_FrameDrops.push_back(len);
				}
				old = curFrame;
			}
			else {
				m_DroppedFrames.push_back(frm);
			}
			frm++;
		}
	}

	// Now copy into the final TOC entries with data
	map<int, TFrameInfo>::iterator pTempToc;

	m_Toc.clear();
	for (pTempToc = tempToc.begin(); pTempToc != tempToc.end(); pTempToc++) {
		if ( pTempToc->second.numChan > 0 ) m_Toc[pTempToc->first] = pTempToc->second.ofs;
	}


	m_HaveToc = true;

	if ( writeTocFile ) {
		string nm = m_Filename + ".toc";
		if ( WriteTocFile(nm) == false && forceFileCreate ) {
			sprintf(buf, "Could not build toc on file %s: %s", nm.c_str(), strerror(errno));
			m_LastError = buf;
			return false;
		}
	}

	m_LastError = "";
	return true;
}


///////////////////////////////////////////////////////////////////////////////
///
/// This function will read data for a set of channels from the DAQ file
/// into memory buffers.  The function does not require a table of contents
/// to operate.
///
/// The input vectors specify which channels to read.  The first vector
/// simply lists the channel identifiers to be read, and the second 
/// vector contains the buffers to receive the corresponding data.
/// Both vectors should be the same size, unless the pointer to the 
/// first vector is 0 which implies the whole file should be read.  In that
/// case, the second vector should be sized large enough to receive
/// all channels.
///
/// The function returns true to indicate all data was read, or false to
/// indicate some problem occurred.
///
bool 
CDaqLowLevelIo::LowLevelDataRead(
	const TIntVec*      pWhichChannels,  
	TDataVec&			userStorage,
	vector<TIntVec>*    pFrameList
)
{
	//
	// Standard error checking
	//
	if ( m_pFile == 0 ) {
		m_LastError = "uninitialized class instance";
		return false;
	}

	bool haveChansBadSize;
	bool noChansBadSize;

	haveChansBadSize = pWhichChannels && pWhichChannels->size() != userStorage.size();
	noChansBadSize   = pWhichChannels == 0 && userStorage.size() != m_NumEntries;
	if ( haveChansBadSize || noChansBadSize ) {
		m_LastError = "inconsistently sized input arguments";
		return false;
	}

	char buf[256];              // buffer for writing messages

	// temporary storage; use a vector so we don't have to worry about
	// memory leaks
	vector<unsigned char>  tempSpace;
	tempSpace.resize(cMaxDaqRec);
	void *pTempSpace = (void *)&tempSpace[0];

	//
	// Create a faster data structure to tell us which channels are needed.
	// Once done, needChan[channelId] is -1 if we don't need, or the index
	// within the user specified 'data' vector of where data should be stored
	//
	vector<int>  needChan;
	if ( pWhichChannels ) {
		unsigned     i;

		needChan.resize(m_NumEntries, -1);
		for (i=0; i<pWhichChannels->size(); i++) {
			if ( (*pWhichChannels)[i] < 0 || (*pWhichChannels)[i] >= m_NumEntries) {
				sprintf(buf, "Invalid channel %d", (*pWhichChannels)[i]);
				m_LastError = buf;
				return false;
			}
			needChan[(*pWhichChannels)[i]] = i;
		}
	}
	else {
		int i;
		for (i=0; i<m_NumEntries; i++)
			needChan.push_back(i);
	}

	TIntVec empty;
	if ( pFrameList ) pFrameList->resize(needChan.size(), empty);
	//
	// Go through the file collecting the data we need.
	//
	int numFrames = 0;
	int curFrame = -1;     // frame for current record
	_fseeki64(m_pFile, m_DataOffset, SEEK_SET);
	while ( 1 ) {
		int  recHead[3];        // (Code, Frame, Count)
		int  numChan;           // number of channels in current record

		// If we have an end frame, use it as a loop termination criteria
		if ( m_EofStatus != eDAQ_EOF_NOTFOUND && m_EofStatus != eDAQ_EOF_UNDEFINED ) {
			if ( curFrame >= m_LastFrame ) break;
		}

		if ( fread(recHead, sizeof(int), 3, m_pFile) != 3 ) {
			sprintf(buf, "Couldn't read file, premature end or messed up file");
			m_LastError = buf;
			return false;
		}

		if ( recHead[0] == -1 ) {           // This is a frame marker
			curFrame  = recHead[1];
			numChan   = recHead[2];
			numFrames++;

			// print info every 30 seconds of collection (30*240=720)
			if ( (numFrames % 720) == 0 ) 
				(*m_UserProgrFunc)(m_Filename.c_str(), numFrames, m_LastFrame - m_FirstFrame);
		}
		else if ( recHead[0] == -2 ) {      // This is an end marker
			curFrame = recHead[1];
			if ( recHead[2] == DAQ_END_MARK ) {
				break; // successfully found EOF
			}
			else {
				sprintf(buf, "Unexpected secondary code %d at line %d", recHead[0], 
						__LINE__);
				m_LastError = buf;
				return false;
			}
		}
		else {
			sprintf(buf, "Unexpected header code %d at file offset %I64d", 
					recHead[0], _ftelli64(m_pFile));
			(*m_UserErrorFunc)(m_Filename.c_str(), buf);

			long long skip;
			long long ofs = _ftelli64(m_pFile);
			if ( ResyncFrame(curFrame, skip) == false ) {
				sprintf(buf, "Could not recover after error at file offset %I64d", ofs);
				m_LastError = buf;
				return false;
			}
			else {
				continue;
			}
		}

		// Now read the data for this record
		int idx;
		for (idx=0; idx<numChan; idx++) {
			int chId;

			if ( fread(&chId, sizeof(int), 1, m_pFile) != 1 ) {
				sprintf(buf, "Couldn't read file, premature end, frm=%d, ofs=%d", curFrame, ftell(m_pFile));
				m_LastError = buf;
				return false;
			}
			if ( chId >= m_NumEntries || chId < 0 ) {
				long long dummy;
				int rcode = ResyncChannel(curFrame, dummy);
				if ( rcode == eRESYNC_CHANNEL || rcode == eRESYNC_FRAME ) {
					printf("ERROR, ofs %d\n", ftell(m_pFile));
					break;
				}
				sprintf(buf, "Encountered invalid channel %d in data area", chId);
				m_LastError = buf;
				return false;
			}
			CDaqChannelInfo& ch = m_Channels[chId];
			int size;

			if ( ch.IsVarLen() ) {
				int actCount;

				if ( fread(&actCount, sizeof(int), 1, m_pFile) != 1 ) {
					sprintf(buf, "Could not read var size channel length; chId=%d, ofs=%I64d, abort",
						chId, _ftelli64(m_pFile));
					m_LastError = buf;
					(*m_UserErrorFunc)(m_Filename.c_str(), buf);
					return false;
				}
				size = ch.GetItemSize() * actCount;
			}
			else {
				size = ch.GetRecSize();
			}
			if ( needChan[chId] != -1 ) {	// it is -1 when we don't need it
				CDaqBuffer&  daqBuf = userStorage[needChan[chId]];

				if ( size > cMaxDaqRec ) {
					sprintf(buf, "Daq record size too big, increase \"cMaxDaqRec\"");
					m_LastError = buf;
					return false;
				}

				if ( size > 0 ) {
					size_t n = fread(pTempSpace, size, 1, m_pFile);
					if ( n != 1 ) {
						sprintf(buf, "read data for channel %d near frame %d failed",
						chId, curFrame);
						m_LastError = buf;
						return false;
					}
				}

				// for variable size records, 'size' may be less than the full size for the channel
				// The shorter storage used in the DAQ is transparent to the user so we fill
				// the rest of the space (that didn't get read from the file with zeros.
				int fullSize = ch.GetRecSize();
				if ( fullSize != size && size < fullSize ) {
					memset(((char *)pTempSpace) + size, 0, fullSize - size);
				}
				daqBuf.Append(pTempSpace, fullSize, ch.GetItemCount());
				if ( pFrameList && (int)pFrameList->size() > needChan[chId]) 
				{
					(*pFrameList)[needChan[chId]].push_back(curFrame);
				}
			}
			else {
				// skip the data
				if ( size ) fseek(m_pFile, size, SEEK_CUR);
			}
		}
	}

	return true;
}


///////////////////////////////////////////////////////////////////////////////////////////
///
/// This function copies  data from a DAQ file into a set of output files for which the 
/// user has already obtained FILE handles.  The function can optionally fill in
/// missing data and provides options on how to handle differentially stored data.
///
/// The function will scan the whole file and store the samples of each channel 
/// specified in chans into the corresponding file. The options argument controls 
/// what the function does when it encounters dropped frames or differentially 
/// stored channels. By default, dropped frames are ignored and only samples in 
/// the DAQ file are written to the output, even for differential data.
///
/// If the eEXPAND_DIFFERENTIAL option is set, then when encountering a differential 
/// channel, the function will write out the data at the maximum sampling frequency 
/// by using the most recent value for samples that were intentionally skipped.
///
/// If the eFILL_MISSING option is set, then the function will detect dropped 
/// frames and write out replacement values to ensure that all the output file(s) 
/// contain the appropriate number of samples. Data for dropped frames is generated 
/// by using the most recent data of the same channel.
///
/// The output files can be written in ASCII or binary.  The argument FileHandles
/// is a vector of FILE pointers that should already have been opened successfully.
/// The file handles must match the binary/ascii option specified in options. 
/// When asking for data in ASCII, 
/// each sample will produce a line in the output file. That line contains the 
/// items for the sample separated by a space. When asking for data in binary, 
/// the data is written in raw binary format. For example, a channel with 3 
/// floats per sample will occupy 12 bytes per sample.
///
/// The function will not close the file handles before returning, that is the
/// responsibility of the caller.
///
/// The final option controlling the data written to the file is the 
/// eINCLUDE_FRAME option. If this option is specified, a frame number is 
/// written before each sample in the output file. In ASCII mode, the frame is 
/// written as an integer at the beginning of each line and in binary mode a 
/// four byte integer is written before each sample.
///
/// The function returns true to indicate that it successfully went through 
/// the file and wrote the data, or false if it encountered an unrecoverable error. 
///
bool 
CDaqLowLevelIo::LowLevelDataCopy(
	const TIntVec*  pWhichChannels,	///< List of channels to write to the output
	TFileVec&       FileHandles,	///< Open file handles to use for corresponding channels
	TConvertOptions options			///< Options controlling how data is written
)
{
	//
	// Standard error checking
	//
	if ( m_pFile == 0 ) {
		m_LastError = "uninitialized class instance";
		return false;
	}

	if ( m_HaveToc == false ) {
		m_LastError = "need to build table of contents first";
		return false;
	}

	if ( pWhichChannels && pWhichChannels->size() != FileHandles.size() ) {
		m_LastError = "inconsistently sized input arguments";
		return false;
	}

	char buf[256];              // buffer for writing messages

	// temporary storage; use a vector so we don't have to worry about
	// memory leaks
	vector<unsigned char>  tempSpace;
	tempSpace.resize(cMaxDaqRec);
	void *pTempSpace = (void *)&tempSpace[0];
	vector<CDaqBuffer>   latestData;		// latest sample of data

	//
	// Create a faster data structure to tell us which channels are needed.
	// Once done, needChan[channelId] is -1 if we don't need, or the index
	// within the user specified 'data' vector of where data should be stored
	//
	vector<int>  needChan;

	if ( pWhichChannels ) {
		unsigned     i;
		CDaqBuffer   temBuf;

		needChan.resize(m_NumEntries, -1);
		latestData.resize(m_NumEntries, temBuf);

		for (i=0; i<pWhichChannels->size(); i++) {
			if ( (*pWhichChannels)[i] < 0 || (*pWhichChannels)[i] >= m_NumEntries) {
				return false;
			}
			needChan[(*pWhichChannels)[i]] = i;
		}
	}
	else {
		int i;
		CDaqBuffer   temBuf;

		latestData.resize(m_NumEntries, temBuf);
		for (i=0; i<m_NumEntries; i++)
			needChan.push_back(i);
	}

	int    frm = m_FirstFrame;
	bool   NoExpandDiff = (options & eEXPAND_DIFFERENTIAL) == 0;
	vector<bool> gotData;

	gotData.resize(m_NumEntries, false);
	while ( frm <= m_LastFrame ) {
        TTocMap::const_iterator        pTocEn;	// iterates list of frames for which we have data
		vector<int>::const_iterator    pDrpFrm;	// iterates list of dropped frames
		bool                           curFrameLost = false;
		int	                           numChan;

		// determine if this is a frame that was dropped by searching through the
		// list of dropped frames calculated during the integrity verification phase
		for (pDrpFrm = m_DroppedFrames.begin(); pDrpFrm != m_DroppedFrames.end(); pDrpFrm++){
			if ( frm == *pDrpFrm ) {
				curFrameLost = true;
				break;
			}
		}

		if ( curFrameLost == false ) {
			pTocEn = m_Toc.find(frm);	// does it have data ?
			if ( pTocEn == m_Toc.end() ) {	// no data collected for this frame
				// do nothing
			}
			else {
				int actualFrameRead;
				long long dummy;
				TReadErrorCode rcode;

				rcode = ReadFrameHeader(frm, actualFrameRead, numChan, dummy);
				if ( rcode != eDAQ_READ_OK ) {
					sprintf(buf, "ReadFrameHeader returned %d",  rcode);
					m_LastError = buf;
					return false;
				}
				assert(frm == actualFrameRead);

				for (int ii=0; ii<m_NumEntries; ii++) gotData[ii] = false;
				ReadDataForOneFrame(numChan, needChan, latestData, pTempSpace, gotData);

				for (int ch=0; ch < m_NumEntries; ch++) {
					CDaqChannelInfo& chInf = m_Channels[ch];
					int              size  = chInf.GetRecSize();
					size_t           out;
					
					int nn = needChan[ch];

					if ( options & eASCII_FORMAT ) {
						WriteAsciiLine(FileHandles[needChan[ch]],chInf,
							latestData[needChan[ch]].GetDataPtr(),  options);
					}
					else {		// binary format
						// if this is a differentially stored channel, we must
						// take into account the user option
						bool NoNewData    = latestData[needChan[ch]].GetSize() == 0;
						bool differential = chInf.GetCapRate() < 0;

						if ( differential && NoExpandDiff && NoNewData )
							continue;

						if ( options & eINCLUDE_FRAME ) {
							fwrite(&frm, sizeof(int), 1, FileHandles[needChan[ch]]);
						}
						out = fwrite(latestData[needChan[ch]].GetDataPtr(), size, 1, 
									FileHandles[needChan[ch]]);
						if ( out != 1 ) {
							sprintf(buf, "Write error while outputing channel %d", ch);
							m_LastError = buf;
							return false;
						}
					}
				}
			}
		}
		else {	// current frame lost
			if ( options & eFILL_MISSING ) {
				// Produce data for current frame based on prior values
				for (int ch=0; ch < m_NumEntries; ch++) {
					CDaqChannelInfo& chInf = m_Channels[ch];
					int              size  = chInf.GetRecSize();
					size_t           out;

					bool NoNewData    = latestData[needChan[ch]].GetSize() == 0;
					bool differential = chInf.GetCapRate() < 0;

					if ( differential && NoExpandDiff && NoNewData )
						continue;

					if ( options & eASCII_FORMAT ) {

						WriteAsciiLine(FileHandles[needChan[ch]], chInf, 
							latestData[needChan[ch]].GetDataPtr(), options);
					}
					else {
						if ( options & eINCLUDE_FRAME ) {
							fwrite(&frm, sizeof(int), 1, FileHandles[needChan[ch]]);
						}
						out = fwrite(latestData[needChan[ch]].GetDataPtr(), size, 1, 
							FileHandles[needChan[ch]]);
						if ( out != 1 ) {
							sprintf(buf, "Write error while outputing channel %d", ch);
							m_LastError = buf;
							return false;
						}
					}
				}
			}
		}
		frm++;
	}
	return true;
}


///////////////////////////////////////////////////////////////////////////////////////////
///
/// This function is similar to LowLevelDataRead but provides additional options for 
/// data management and selection of a range of frames to read. Also, unlike 
/// LowLevelDataRead, this function can only be called if the class has access
/// to the table of contents (either built with the CheckDataIntegrity function or
/// read from a prior run). Finally, the function can expand differentially 
/// stored channels or write them out in compressed format, according to user options
///
/// The function reads the specified list of channels into memory buffers, 
/// expands differentially stored data, and fills in data for dropped frames.
/// If specified, it can read a subset of the data based on a start and end frame.
///
/// The pWhichChannels vector lists the identifiers for which channels to read. 
///
/// The userDataStorage argument is a vector of CDaqBuffer classes, each of which
/// will contain the data for a channel.  The vector can be empty, in which case
/// the function will expand it to be equal to the number of requested channels.
/// If the vector is already properly sized, the function will simply put
/// the data on the corresponding entry.  For example, the first requested channel will
/// be stored in userDataStorage[0], the second requested channel will be stored
/// in userDataStorage[1] etc.
///
/// Finally, note that for differentially stored data, the extrapolation 
/// method defaults to zero-hold.
///
/// The frm1 and frm2 arguments specify the first/last fbs frame to read, inclusively.
/// Specify -1 to read from the start or to the end of file respectively.
///
bool 
CDaqLowLevelIo::GetFullData(
	const TIntVec*  pWhichChannels,      
	TDataVec&       userDataStorage,	///< Where to store the data
	TConvertOptions option,
	int             frm1,				///< First frame to sample, or -1 for first in file
	int             frm2,				///< Last frame to sample, or -1 for last in file
	vector<TIntVec>* pFrmList,
	int maxDiffLook)			///< Where to store frame sequences
{
	char buf[256];						// buffer for writing messages

	//
	// Standard error checking
	//
	if ( m_pFile == 0 ) {
		m_LastError = "uninitialized class instance";
		return false;
	}
	if ( m_HaveToc == false ) {
		m_LastError = "need to buld table of contents first";
		return false;
	}
	if ( (option & eINCLUDE_FRAME) && pFrmList == 0 ) {
		m_LastError = "must provide storage for frames when eINCLUDE_FRAME is specified";
		return false;
	}

	//
	// Create a faster data structure to tell us which channels are needed.
	// Once done, needChan[channelId] is -1 if we don't need, or the index
	// within the user specified 'data' vector of where data should be stored
	//
	vector<int>  needChan;
	if ( pWhichChannels ) {
		if ( pWhichChannels->size() == 0 ) return true;
		needChan.resize(m_NumEntries, -1);
		for (unsigned i=0; i<pWhichChannels->size(); i++) {
			if ( (*pWhichChannels)[i] < 0 || (*pWhichChannels)[i] >= m_NumEntries) {
				sprintf(buf, "invalid channel id %d ", (*pWhichChannels)[i]);
				m_LastError = buf;
				return false;
			}
			needChan[(*pWhichChannels)[i]] = i;
		}
	}
	else {
		for (int i=0; i<m_NumEntries; i++) needChan.push_back(i);
	}

	vector<bool>        gotData;					// keep track of which channel has data
	const vector<int>*  pChnList;
	vector<int>         fullList;

	gotData.resize(m_NumEntries, false);

	if ( pWhichChannels ) {
		pChnList = pWhichChannels;
	}
	else {
		fullList.resize(m_NumEntries);
		for (int idx=0; idx < m_NumEntries; idx++) fullList[idx] = idx;
		pChnList = &fullList;
	}

	// clear any buffers that the user has provided
	for (unsigned ii=0; ii<userDataStorage.size(); ii++) {
		userDataStorage[ii].Clear();
	}

	// make sure the storage vectory is properly dimensioned and initialized
	if ( pChnList->size() > userDataStorage.size() ) {
		CDaqBuffer empty;
		userDataStorage.resize(pChnList->size(), empty);
	}

	// temporary storage; use a vector so we don't have to worry about
	// memory leaks
	vector<unsigned char>  tempSpace;
	tempSpace.resize(cMaxDaqRec);
	void *pTempSpace = (void *)&tempSpace[0];

	//
	// Various other variables
	//
	vector<CDaqBuffer>   latestData;		// latest sample of data
	CDaqBuffer           temBuf;			// empty buffer to be used as template for latestData
	int                  curFrame;
	int                  numChan;           // number of channels in current record
	long long            bytesSkipped;
	TIntVec              empty;

	if ( pFrmList) pFrmList->resize(userDataStorage.size(), empty);
	latestData.resize(pChnList->size(), temBuf);
	CDaqLowLevelIo::TReadErrorCode rcode;

	// 
	// Adjust frame numbers if the user specified -1.  Also, if frames are specified,
	// adjust them to make sure they refer to frames that contain data; we look ahead for 
	// the first frame and we look backwards for the last frame, that way we give 
	// the user slightly less than what was asked.
	//
	if ( frm1 < 0 ) frm1 = GetFirstFrame(); 
	if ( frm2 < 0 ) frm2 = GetLastFrame();

	if ( frm1 < GetFirstFrame() || frm1 > GetLastFrame() 
				|| frm2 < GetFirstFrame() || frm2 > GetLastFrame() ) {
		m_LastError = "invalid frame specification; remember frames are fbs based";
		return false;
	}

	int triesLeft = 32;
	while ( m_Toc.find(frm1) == m_Toc.end() && triesLeft-- > 0 ) frm1++;
    if (triesLeft < 0) return false;
	assert(triesLeft>0);		// shouldn't really happen

	triesLeft = 32;
	while ( m_Toc.find(frm2) == m_Toc.end() && triesLeft-- > 0 ) frm2--;
	assert(triesLeft>0);		// shouldn't really happen

	//
	// If the first frame to read is not the first one in the file, and if
	// there are differential channels, we must find the most recent value
	// of all differential channels, otherwise we may need to output data nad
	// have none
	//
	if ( frm1 != GetFirstFrame() ) {
		int endFrame = GetFirstFrame();
		if (maxDiffLook>0){
			endFrame = max(frm1 - maxDiffLook,endFrame);
		}
		for (size_t idx=0; idx < pChnList->size(); idx++) {
			if ( m_Channels[(*pChnList)[idx]].GetCapRate() < 0 ) {
				// Search backwards from frame frm1 until
				// we find data for the specific channel.  Store
				// its values in latestData[idx]
				bool done      = false;
				int  candFrame = frm1;
				int  theChan   = (*pChnList)[idx];

				while ( !done && candFrame >= endFrame ) {
					ReadFrameHeader(candFrame, curFrame, numChan, bytesSkipped);
					ReadChannelDataForOneFrame(numChan, theChan, latestData[idx], pTempSpace, done);
					candFrame--;
				}
				if (candFrame < endFrame){
					done = true;
				}
				if ( !done && endFrame < 0 ) {
					m_LastError = "could not find any data for var channel; possible internal error";
					return false;
				}
			}
		}
	}

	// Seek to the first frame we need
	rcode = ReadFrameHeader(frm1, curFrame, numChan, bytesSkipped);
	if ( rcode != eDAQ_READ_OK ) {
		sprintf(buf, "First ReadFrameHeader(frm1=%d) returned %d",  frm1, rcode);
		m_LastError = buf;
		return false;
	}
	if (curFrame>frm2){ //DAH
		frm2=curFrame;
	}
	while ( curFrame <= frm2 ) {
		bool curFrameLost = FrameDropped(curFrame);

		if ( curFrameLost ) {
			if ( option & eFILL_MISSING ) {
				// Re-use the most recent data
				for (size_t idx=0; idx < pChnList->size(); idx++) {
					CDaqChannelInfo& chInf = m_Channels[(*pChnList)[idx]];
					int              size  = chInf.GetRecSize();

					userDataStorage[idx].Append(latestData[idx].GetDataPtr(), size,
						chInf.GetItemCount());
					if ( pFrmList ) (*pFrmList)[idx].push_back(curFrame);
				}
			}
		}
		else {
			if ( numChan > 0 ) {
				for (int ii=0; ii < m_NumEntries; ii++) gotData[ii] = false;

				ReadDataForOneFrame(numChan, needChan, latestData, pTempSpace, gotData);
				for (size_t idx=0; idx < pChnList->size(); idx++) {
					CDaqChannelInfo& chInf    = m_Channels[(*pChnList)[idx]];
					int              size     = chInf.GetRecSize();
					bool             diffChan = chInf.GetCapRate() < 0;
					const void*      pSrc = latestData[idx].GetDataPtr();

					if (pSrc) {
						if ( diffChan && (option & eEXPAND_DIFFERENTIAL) ) {
							userDataStorage[idx].Append(pSrc, size, chInf.GetItemCount());
							if ( pFrmList) (*pFrmList)[idx].push_back(curFrame);
						}
						else if ( !diffChan && gotData[(*pChnList)[idx]] ) {
							userDataStorage[idx].Append(pSrc, size, chInf.GetItemCount());
							if ( pFrmList) (*pFrmList)[idx].push_back(curFrame);
						}
					}
				}
			}
			else {		// no data for that frame
				if ( option & eEXPAND_DIFFERENTIAL ) {
					// when expanding differential data, we output samples for each frame
					for (size_t idx=0; idx < pChnList->size(); idx++) {
						int chId = (*pChnList)[idx];

						if ( m_Channels[chId].GetCapRate() < 0 ) {
							int         size = m_Channels[chId].GetRecSize();
							const void* pSrc = latestData[idx].GetDataPtr();
							userDataStorage[idx].Append(pSrc, size, m_Channels[chId].GetItemCount());
							if ( pFrmList ) (*pFrmList) [idx].push_back(curFrame);
						}
					}
				}
				else {
                    // Do nothing
				}
			}
		}
		rcode = ReadFrameHeader(-1, curFrame, numChan, bytesSkipped);
		if ( rcode != eDAQ_READ_OK && rcode != eDAQ_READ_EOF) {
			sprintf(buf, "ReadFrameHeader returned %d (curFrame=%d, last=%d)",  rcode, curFrame, frm2);
			m_LastError = buf;
			//return false;
			break;
		}
	}

	return true;
}


///////////////////////////////////////////////////////////////////////////////
//
// This function reads channel data for the current record.  It assumes that
// the file pointer points to the first record immediately after the
// frame header.  It will read as many samples as indicated in the 'numEntries'
// input argument.  The needChannel array indicates if a given channel id
// is needed or not.  It is dimensioned according to the total number of
// channels available in the DAQ file.
// The storage array holds the data for this frame, one entry per channel. 
// Note that indexing the needChannel array is different than the storage
// array, in particular, needChannel should be m_NumEntries long but storage
// is dimensioned long enough to keep track of the needed entires as specified
// in needChannel.  
//
// As an example, assume that the number of channels in the file is 20 (0..19).
// The value of m_NumEntries should be 20 also.  The array needChannel should
// be 20 ints long.  It will look something like this:
// -1 -1 -1 0 -1 1 -1 -1 2 3 4 -1 -1 -1 -1 5 6 7 -1 -1
// indicating that we care about channels 3, 5, 8, 9, 10, 15, 16, and 17.
// The storage array should be 8 in length, with channel 3 in position 0,
// channel 5 in position 1, channel 8 in position 2, channel 9 in position 3 etc.
// In effect, if needChannel[i] is >=0, its data should be stored
// in storage[needChannel[i]]
// 
// Finally, pTempSpace is a buffer where the function can temporarily read
// values.  We pass it down to avoid repeated memory allocations.
//
CDaqLowLevelIo::TReadErrorCode
CDaqLowLevelIo::ReadDataForOneFrame(
	int           numEntriesInThisRecord,
	TIntVec       needChannel,
	TDataVec&     storage,
	void*         pTempSpace,
	vector<bool>& gotData
)
{
	int idx;
	char buf[256];

	for (idx=0; idx<numEntriesInThisRecord; idx++) {
		int chId;
		if (m_pFile == NULL) return eDAQ_READ_ERROR;

		if ( fread(&chId, sizeof(int), 1, m_pFile) != 1 ) return eDAQ_READ_ERROR;

		if ( chId >= m_NumEntries || chId < 0 ) {
			sprintf(buf, "Encountered invalid channel %d in data area", chId);
			m_LastError = buf;
			return eDAQ_READ_BAD_DATA;
		}

		CDaqChannelInfo& ch   = m_Channels[chId];
		size_t         size;

		if ( ch.IsVarLen() ) {
			int actCount;
			if ( fread(&actCount, sizeof(int), 1, m_pFile) != 1 ) {
				sprintf(buf, "Could not read var size channel length; chId=%d, ofs=%I64d, abort",
					chId, _ftelli64(m_pFile));
				m_LastError = buf;
				(*m_UserErrorFunc)(m_Filename.c_str(), buf);
				return eDAQ_READ_ERROR;
			}
			size = ch.GetItemSize() * actCount;
		}
		else {
			size = ch.GetRecSize();
		}

		if ( needChannel[chId] != -1 ) {	// it is -1 when we don't need it
			CDaqBuffer&  daqBuf = storage[needChannel[chId]];

			if ( size > cMaxDaqRec ) {
				sprintf(buf, "Daq record size too big, increase \"cMaxDaqRec\"");
				m_LastError = buf;
				return eDAQ_READ_INTERNAL_ERROR;
			}

			if ( size > 0 ) {
				size_t n = fread(pTempSpace, size, 1, m_pFile);
				if ( n != 1 ) {
					sprintf(buf, "read data for channel %d failed", chId);
					m_LastError = buf;
					return eDAQ_READ_ERROR;
				}
			}

			int fullSize = ch.GetRecSize();
			if (size < fullSize){
				size = fullSize;
			}
			if ( fullSize != size && size < fullSize ) {
				memset(((char *)pTempSpace) + size, 0, fullSize - size);
			}
			daqBuf.Replace(pTempSpace, fullSize, ch.GetItemCount());
			gotData[chId] = true;
		}
		else {
			// skip the data
            _fseeki64(m_pFile, size, SEEK_CUR);
		}
	}
	return eDAQ_READ_OK;
}

///////////////////////////////////////////////////////////////////////////////
//
// This function reads data for a specific channel from the current frame
//
CDaqLowLevelIo::TReadErrorCode
CDaqLowLevelIo::ReadChannelDataForOneFrame(
	int           numEntriesInThisRecord,
	int           channel,
	CDaqBuffer&   daqBuf,
	void*         pTempSpace,
	bool&         gotData
)
{
	int idx;
	char buf[256];

	for (idx=0; idx<numEntriesInThisRecord; idx++) {
		int chId;
		if ( fread(&chId, sizeof(int), 1, m_pFile) != 1 ) return eDAQ_READ_ERROR;

		if ( chId >= m_NumEntries || chId < 0 ) {
			sprintf(buf, "Encountered invalid channel %d in data area", chId);
			m_LastError = buf;
			return eDAQ_READ_BAD_DATA;
		}

		CDaqChannelInfo& ch   = m_Channels[chId];
		int              size;

		if ( ch.IsVarLen() ) {
			int actCount;
			if ( fread(&actCount, sizeof(int), 1, m_pFile) != 1 ) {
				sprintf(buf, "Could not read var size channel length; chId=%d, ofs=%I64d, abort",
					chId, _ftelli64(m_pFile));
				m_LastError = buf;
				(*m_UserErrorFunc)(m_Filename.c_str(), buf);
				return eDAQ_READ_ERROR;
			}
			size = ch.GetItemSize() * actCount;
		}
		else {
			size = ch.GetRecSize();
		}

		if ( chId == channel ) {	// we only care for the particular channel
			if ( size > cMaxDaqRec ) {
				sprintf(buf, "Daq record size too big, increase \"cMaxDaqRec\"");
				m_LastError = buf;
				return eDAQ_READ_INTERNAL_ERROR;
			}

			if ( size > 0 ) {
				size_t n = fread(pTempSpace, size, 1, m_pFile);
				if ( n != 1 ) {
					sprintf(buf, "read data for channel %d failed", chId);
					m_LastError = buf;
					return eDAQ_READ_ERROR;
				}
			}

			int fullSize = ch.GetRecSize();
			if ( fullSize != size && size < fullSize ) {
				memset(((char *)pTempSpace) + size, 0, fullSize - size);
			}
			daqBuf.Replace(pTempSpace, fullSize, ch.GetItemCount());
			gotData = true;
			return eDAQ_READ_OK;
		}
		else {
			// skip the data
			fseek(m_pFile, size, SEEK_CUR);
		}
	}
	return eDAQ_READ_OK;
}


/////////////////////////////////////////////////////////////////////////////
///
/// This function reads the header for the upcoming frame.  It will verify
/// it's a correct header and try to re-sync reading if the next bytes do
/// not make up a reasonable header.
///
/// The frameToRead argument indicates which frame we are trying to
/// read.  If not set to -1, the function will assume a table of contents
/// exists and will seek to the specific offset within the file, but if
/// frameToRead is set to -1, the function simply assumes the file 
/// pointer is set to read the header next.
///
/// Upon a successful return, curFrame is the fbs frame number that
/// was read and numChan is the number of channels stored for that frame.
///
/// The skip variable indicates how many bytes had to be skipped in
/// order to find a good header.  U
///
///\return OK, EOF, SKIPPED, ERROR
///
CDaqLowLevelIo::TReadErrorCode
CDaqLowLevelIo::ReadFrameHeader(
	int     frameToRead,// which frame we anticipate reading, use -1 to read next available 
	int&    curFrame,	// which frame was read
	int&    numChan,	// number of channels in this frame
	long long&    skip		// how many bytes were skipped (if >0 indicates some corruption)
)
{
	int    recHead[3];        // (Code, Frame, Count)
	char   buf[256];          // buffer for writing messages
	
	skip = 0;	// by default, we expect no skipping

	// We run a loop because we may have to try again in case of frame
	// drops or other file corruption
	for ( ; ; ) {
		if ( frameToRead >= 0  ) {
			long long startOfs;
            TTocMap::const_iterator	mapIter;

			mapIter = m_Toc.find(frameToRead);
			if ( mapIter == m_Toc.end() ) {
				return eDAQ_NO_SUCH_FRAME;
			}
			startOfs = mapIter->second;
			_fseeki64(m_pFile, startOfs, SEEK_SET);
		}

		// Read the frame header; if this fails, we are really in trouble
		if ( fread(recHead, sizeof(int), 3, m_pFile) != 3 ) {
			if (feof(m_pFile) != 0){
				sprintf(buf, "Couldn't read file, premature end of file");
				
			}else if (ferror(m_pFile) != 0){
				sprintf(buf, "Couldn't read file, system read error");
			}else{
				sprintf(buf, "Couldn't read file, uknown issue, EOF not set, read error flag not set");
			}
			m_LastError = buf;
			return eDAQ_READ_ERROR;
		}

		// Most of the time, this is what we'll get
		if ( recHead[0] == -1 ) {  // Frame marker 
			curFrame  = recHead[1];
			numChan   = recHead[2];

			// print info every 30 seconds of collection (30*240=720)
			if ( (curFrame % 720) == 0 ) {
				(*m_UserProgrFunc)(m_Filename.c_str(), curFrame - m_FirstFrame, 
					m_LastFrame - m_FirstFrame);
			}
			return eDAQ_READ_OK;
		}

		if ( recHead[0] == -2 ) {      // This is an end marker, so we're done
			curFrame = recHead[1];
			if ( recHead[2] != DAQ_END_MARK ) {
				// problem
			}
			return eDAQ_READ_EOF;
		}
		// If we get here, then what we expected to be a frame header is not.  
		// Most likely some data didn't make it and we are out of sync with
		// where things are.  We will try to resynchronize by looking for
		// a reasonable frame marker at some point ahead 
		sprintf(buf, "Expected header code (-1) but found %d at fileofs %I64d", 
					recHead[0], _ftelli64(m_pFile));
		(*m_UserErrorFunc)(m_Filename.c_str(), buf);

		auto ofs = _ftelli64(m_pFile);
		if ( ResyncFrame(curFrame, skip) == false ) {
			sprintf(buf, "Could not recover after error at file offset %I64d", ofs);
			m_LastError = buf;
			return eDAQ_READ_FAILED;
		}
		frameToRead = -1;		// no seeking needed after a resync
	}
}
#ifdef WIN32
    #pragma warning(pop)
#endif
/// @}
