/////////////////////////////////////////////////////////////////////////////
//
// DAQ low level IO Library
// 
//
#include "DaqIo.h"


CDaqLowLevelIo::CDaqLowLevelIo()
{
}

CDaqLowLevelIo::~CDaqLowLevelIo()
{
}


void
NearDump(FILE* p)
{
	printf("\nData dump near error\n");
	fseek(p, -8, SEEK_CUR);
	int d;

	for (int i=0; i<8; i++) {
		size_t n = fread(&d, sizeof(d), 1, p);
		printf("%-d:  %d (code=%d)\n", ftell(p)-sizeof(d), d, n);
	}
}

int 
DataSize(CDaqChannelInfo& ch)
{
	int size;

	switch ( ch.m_Type ) {
		case 'f' : 
		case 'i' : size = 4; break;

		case 'd' : size = 8; break;

		case 's' : size = 2; break;
		case 'c' : size = 1; break;
		default : return -1;
	}

	return size * ch.m_Items;
}

 
/////////////////////////////////////////////////////////////////////////////
///
/// Reads the header of the DAQ file, independent of its version, and 
/// stores the information in variables local to the class instance.  
/// Values that are not provided in the specific file version are set
/// to some default
///
bool
CDaqLowLevelIo::ReadHeader(	FILE *p)
{
	unsigned magic;

	if ( fread(&magic, sizeof(magic), 1, p) != 1 ) {
		return false;
	}
	rewind(p);

	bool supported = false;

	if  ( magic == DAQ_MAGIC_NUM_VER_2_0 ) {
		printf("Version 2.0\n");
	}
	else if ( magic == DAQ_MAGIC_NUM_VER_2_1 ) {
		printf("Version 2.1\n");
	}
	else if ( magic == DAQ_MAGIC_NUM_VER_2_2 ) {
		printf("Version 2.2\n");
		struct Hdr {
			unsigned Magic;
			char     Title[120];
			char     Date[27];
			char     Subj[128];
			char     Run[128];
			char     RunInst[128];
			int      NumEntries;
			unsigned DAQFrequency;
		} header = { 0 };

		if ( fread(&header, sizeof(header), 1, p) != 1 ) {
			return false;
		}
		m_Title      = header.Title;
		m_Date       = header.Date;
		m_Subj       = header.Subj;
		m_Run        = header.Run;
		m_RunInst    = header.RunInst;
		m_NumEntries = header.NumEntries;
		m_Frequency  = header.DAQFrequency;
		supported = true;
	}

	if ( !supported ) {
		fprintf(stderr, "Unsuppored version, magic=0x%X\n", magic);
		return false;
	}

	return true;
}


/////////////////////////////////////////////////////////////////////////////
///
/// Reads the information about all channels stored in the DAQ file.
/// Dagta is stored in the m_Channels instance variable
///
bool
CDaqLowLevelIo::ReadChannelInfo(FILE *p)
{
	int     ch;

    struct ChInfo {
      int   ItemCount;
      char  Name[36];
      char  Units[16];
      short CaptureRate;
      int   DataType;
      char  VarLenFlag;
    } chan;


	for (ch=0; ch < m_NumEntries; ch++) {
		CDaqChannelInfo  chInfo;

		if ( fread(&chan, sizeof(chan), 1, p) != 1 ) {
			return false;
		}

		chInfo.m_Id      = ch;
		chInfo.m_Items   = chan.ItemCount;
		chInfo.m_Name    = chan.Name;
		chInfo.m_Type    = chan.DataType;
		chInfo.m_CapRate = chan.CaptureRate;
		chInfo.m_VarLen  = chan.VarLenFlag ? true : false;

		m_Channels.push_back(chInfo);
	}

	return true;
}


bool
CDaqLowLevelIo::Open(
	const string &  fname)
{

	// 
	// Open the file
	//
	FILE *p = fopen(fname.c_str(), "rb");
	if ( p == 0 ) {
		return false;
	}

	//
	// Read the header and print, if needed
	//
	if ( !ReadHeader(p) ) {
		return false;
	}

	printf("Title     = %s\n", m_Title.c_str());
	printf("Date      = %s", m_Date.c_str());
	printf("Subj      = %s\n", m_Subj.c_str());
	printf("Run       = %s\n", m_Run.c_str());
	printf("RunInst   = %s\n", m_RunInst.c_str());
	printf("Channels  = %d\n", m_NumEntries);
	printf("Frequency = %d\n", m_Frequency);

	//
	// Read channel information and print, if needed
	//
	if ( !ReadChannelInfo(p) ) {
		return false;
	}
	int ch;

	printf("Channels:\n");
	printf("  Id         Name                           Items Type  Rate  VarLen\n");
	for (ch=0; ch<m_NumEntries; ch++) {
		printf("%4d  %-36s %5d    %c     %d    %d\n",
			m_Channels[ch].m_Id, m_Channels[ch].m_Name.c_str(),
			m_Channels[ch].m_Items, m_Channels[ch].m_Type,
			m_Channels[ch].m_CapRate, m_Channels[ch].m_VarLen ? 1 : 0);
	}

	m_DataOffset = ftell(p);

	//
	// Go through the file and build the TOC
	// 
	while ( 1 ) {
		int  three[3];	// (Code, Frame, Count)

		if ( fread(three, sizeof(int), 3, p) != 3 ) {
			fprintf(stderr, "Couldn't read file, premature end or messed up file\n");
			return false;
		}

		if ( three[0] < 0 ) {	// some marker
			if ( three[0] == -1 ) {		// frame
				m_Toc[three[1]] = ftell(p);	// add to TOC
				printf("F=%d ", three[1]);
			}
			else if ( three[0] == -2 ) {	// end marker
				if ( three[2] == DAQ_END_MARK ) {
					break;		/// <-----------  END 
				}
			}
			else {
				fprintf(stderr, "Unexpected code %d at line %d\n", three[0], __LINE__);
				return false;
			}
		}
		else {
			int chId = three[0];
			if ( chId >= m_NumEntries ) {
				fprintf(stderr, "Invalid channel %d\n", chId);
				NearDump(p);
				return false;
			}

			CDaqChannelInfo& ch = m_Channels[chId];
			int  skip = DataSize(ch);
			printf("Channel %d found, %d bytes\n", three[0], skip);

			fseek(p, skip, SEEK_CUR);

		}
	}

	return true;

}
