#include <iostream>
#include <fstream>
#include <DaqIolib.h>
#include <daqplayer.h>
#include <chrono>

#include "NadsDDSLib/NadsDDSLib.h"
using namespace std;
int main(int argc, char **argv) {
    if (argc < 2) {
        cout << "Requires Daq File Name" << endl;
        return 1;
    }

    chrono::high_resolution_clock clock;
    vector<NADSDDS::CPublication::TRef> pubs;
    std::map<string, int> nameToPub;

    std::vector<string> cells;
    try {
        ifstream istream("cells.txt");
        while (!istream.eof() && istream.good()) {
            string temp;
            istream >> temp;
            if (!istream.fail() && temp.size() > 0) {
                cells.push_back(temp);
            }
        }
    }
    catch (std::exception &exp) {
        cout << "Error Reading cell.txt " <<exp.what()<< endl;
        return 1;
    }
    DaqPlayer dp;
    auto ok = dp.OpenDaqFile(argv[1], cells);
    if (!ok) {
        auto error = dp.GetLastError();
        cout << "Got error:" << error << endl;
        return 1;
    }
    for (auto &cell : cells) {
        auto info = dp.GetChannelInfo(cell);
        if (info)
        {          
            auto pub =
                NADSDDS::CPublication::MakePublication(
                    info->GetName(),
                    info->GetId(),
                    info->GetItemCount(),
                    info->GetType()
                );
            nameToPub[info->GetName()] = (int)pubs.size();
            pubs.push_back(std::move(pub));
        }
    }
    auto framePub = NADSDDS::CPublication::MakePublication("Frame", 1, 1, 'i');

    auto start  = clock.now();
    auto startBeat = start.time_since_epoch() / std::chrono::microseconds(16666);
    auto lastBeat = startBeat;
    int cnt = 0;
    while (!dp.Eof()) {
        try {
            auto now  = clock.now();
            auto currentBeat = now.time_since_epoch() / std::chrono::microseconds(16666);
            while (lastBeat>=currentBeat)
            {
                now  = clock.now();
                currentBeat = now.time_since_epoch() / std::chrono::microseconds(16666);
            }
            lastBeat = currentBeat;
            ++dp;
            cnt++;
            if (cnt % 60 == 0)
                cout << "Frame"<<cnt << endl;
        }
        catch (std::exception exp) {
            break;
        }
        for (auto& cell : cells) {
            auto itr = nameToPub.find(cell);
            
            if (itr != nameToPub.end()) {
                
                auto info = dp.GetChannelInfo(cell);
                int maxIndex = std::min((int)pubs[itr->second]->Count(), info->GetItemCount());

                if (info->GetType() == 'i') {
                    auto data = dp.GetData<int>(cell);
                    auto intData = pubs[itr->second]->asInt();
                    int index = 0;
                    
                    for (auto &val : data) {
                        intData[index++] = val;
                        if (index == maxIndex) break;
                    }
                }
                else if (info->GetType() == 's') {
                    auto data = dp.GetData<short>(cell);
                    auto intData = pubs[itr->second]->asShort();
                    int index = 0;
                    for (auto &val : data) {
                        intData[index++] = val;
                        if (index == maxIndex) break;
                    }
                }
                else if (info->GetType() == 'd') {
                    auto data = dp.GetData<double>(cell);
                    auto intData = pubs[itr->second]->asDouble();
                    int index = 0;
                    for (auto &val : data) {
                        intData[index++] = val;
                        if (index == maxIndex) break;
                    }
                }
                else if (info->GetType() == 'f') {
                    auto data = dp.GetData<float>(cell);
                    auto intData = pubs[itr->second]->asFloat();
                    int index = 0;
                    for (auto &val : data) {
                        intData[index++] = val;
                        if (index == maxIndex) break;
                    }
                }
                else if (info->GetType() == 'c') {
                    auto data = dp.GetData<char>(cell);
                    auto intData = pubs[itr->second]->asChar();
                    int index = 0;
                    for (auto &val : data) {
                        intData[index++] = val;
                        if (index == maxIndex) break;
                    }
                }
            }
            
        }
        for (auto &pub : pubs) {
            pub->Publish();
        }
        framePub->asInt()[0] = dp.GetCurrentFrame();
        framePub->Publish();
    }
    return 0;
}